/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2005 by Myricom, Inc.  All rights reserved.                 *
 *************************************************************************/

#include <ndis.h>
#include "mx_arch.h"
#include "mx_instance.h"
#include "mx_malloc.h"
#include "mx_peer.h"
#include "mx_misc.h"
#include "mx_ether_common.h"
#include "mx_stbar.h"
#include "ethernet.h"
#define BYTE_ORDER 0
#define LITTLE_ENDIAN 0
#define BIG_ENDIAN 1
#include "ip.h"

#define MX_ENABLE_ETHERNET 1

unsigned short
in_pseudo(uint32_t a, uint32_t b, uint32_t c);

struct mx_packetq_entry
{
  STAILQ_ENTRY(mx_packetq_entry) entries;
};

STAILQ_HEAD(mx_packetq_head, mx_packetq_entry);

typedef struct mx_adapter
{
  NDIS_HANDLE miniport_adapter_handle;
  PDEVICE_OBJECT pdo;
  ULONG func_num;
  ULONG dev_num;
  ULONG bus_num;

  ULONG filter;
  ULONG lookahead;

  mx_instance_state_t *is;
  NDIS_SPIN_LOCK send_lock;
  NDIS_SPIN_LOCK recv_lock;
  struct mx_packetq_head sendq;

  NDIS_HANDLE recv_small_packet_pool;
  NDIS_HANDLE recv_small_buffer_pool;
  NDIS_HANDLE recv_large_packet_pool;
  NDIS_HANDLE recv_large_buffer_pool;

  ULONG xmit_ok;
  ULONG rcv_ok;
  ULONG xmit_error;
  ULONG rcv_error;
  ULONG rcv_no_buffer;

  ULONG rcv_error_alignment;
  ULONG xmit_one_collision;
  ULONG xmit_more_collisions;

  int do_csum_offload;
  int do_xmit_tcp_csum;
  int do_xmit_udp_csum;
  int do_recv_tcp_csum;
  int do_recv_udp_csum;
} mx_adapter_t;

#define MXE(message) DbgPrint("mxe: " message)

#define MX_TAG ((ULONG)'XM')

#define MX_PACKET_FILTER (NDIS_PACKET_TYPE_BROADCAST | \
                          NDIS_PACKET_TYPE_DIRECTED)

#define MX_LOOKAHEAD 42

/*{{{ oids */
static NDIS_OID g_oids[] = {
  OID_GEN_SUPPORTED_LIST,
  OID_GEN_HARDWARE_STATUS,
  OID_GEN_MEDIA_SUPPORTED,
  OID_GEN_MEDIA_IN_USE,
  OID_GEN_MAXIMUM_LOOKAHEAD,
  OID_GEN_MAXIMUM_FRAME_SIZE,
  OID_GEN_LINK_SPEED,
  OID_GEN_TRANSMIT_BUFFER_SPACE,
  OID_GEN_RECEIVE_BUFFER_SPACE,
  OID_GEN_TRANSMIT_BLOCK_SIZE,
  OID_GEN_RECEIVE_BLOCK_SIZE,
  OID_GEN_VENDOR_ID,
  OID_GEN_VENDOR_DESCRIPTION,
  OID_GEN_VENDOR_DRIVER_VERSION,
  OID_GEN_CURRENT_PACKET_FILTER,
  OID_GEN_CURRENT_LOOKAHEAD,
  OID_GEN_DRIVER_VERSION,
  OID_GEN_MAXIMUM_TOTAL_SIZE,
  OID_GEN_PROTOCOL_OPTIONS,
  OID_GEN_MAC_OPTIONS,
  OID_GEN_MEDIA_CONNECT_STATUS,
  OID_GEN_MAXIMUM_SEND_PACKETS,

  OID_GEN_XMIT_OK,
  OID_GEN_RCV_OK,
  OID_GEN_XMIT_ERROR,
  OID_GEN_RCV_ERROR,
  OID_GEN_RCV_NO_BUFFER,

  OID_802_3_PERMANENT_ADDRESS,
  OID_802_3_CURRENT_ADDRESS,
  OID_802_3_MULTICAST_LIST,
  OID_802_3_MAXIMUM_LIST_SIZE,
  OID_802_3_MAC_OPTIONS,

  OID_802_3_RCV_ERROR_ALIGNMENT,
  OID_802_3_XMIT_ONE_COLLISION,
  OID_802_3_XMIT_MORE_COLLISIONS,

  OID_TCP_TASK_OFFLOAD
};
/*}}}*/

PDEVICE_OBJECT g_mxctl = NULL;
mx_instance_state_t *g_mxctl_is = NULL;
NDIS_HANDLE g_mxctl_handle;

NDIS_HANDLE g_ndis_wrapper;

mx_spinlock_t g_dev_lock;
PDEVICE_OBJECT *g_devices;
mx_instance_state_t **g_instances;
NDIS_HANDLE *g_handles;
mx_endpt_state_t **g_endpts;

void
mx_unit_release(int idx)
{
  ASSERT(g_devices != NULL);
  mx_spin_lock(&g_dev_lock);
  g_devices[idx] = NULL;
  g_handles[idx] = NULL;
  g_instances[idx] = NULL;
  mx_spin_unlock(&g_dev_lock);
}

#define MX_SMALL_PACKET 3
#define MX_BIG_PACKET 7

#if MX_DRIVER_API_MAGIC >=  0x500
#define MX_MUST_DO_4K_ALIGN 1
#else
#define MX_MUST_DO_4K_ALIGN 0
#endif

#if MX_MUST_DO_4K_ALIGN
#define MX_WIN_MAX_ETHER_MTU 1514
#define MX_BIG_LEN 4096
#define MX_BIG_POW2 4096
#else
#define MX_WIN_MAX_ETHER_MTU 9014
#define MX_BIG_LEN 9216
#define MX_BIG_POW2 16384
#endif
#define MX_SMALL_LEN 128

/* Wrappers around memory (de)allocation functions to aid in debugging.
 * Since they aren't in the critical path, no need to condition on
 * DEBUG. */

static unsigned _int64 g_alloc_shared_mem_cnt = 0; /* number of allocations */
static unsigned _int64 g_alloc_shared_mem_len = 0; /* total number of bytes */

VOID
mx__alloc_shared_mem(IN NDIS_HANDLE MiniportAdapterHandle,
		     IN ULONG Length,
		     IN BOOLEAN Cached,
		     OUT PVOID *VirtualAddress,
		     OUT PNDIS_PHYSICAL_ADDRESS PhysicalAddress)
{
  NdisMAllocateSharedMemory(MiniportAdapterHandle,
			    Length,
			    Cached,
			    VirtualAddress,
			    PhysicalAddress);
  if (*VirtualAddress != NULL) {
    g_alloc_shared_mem_cnt++;
    g_alloc_shared_mem_len += Length;
  }
}

VOID
mx__free_shared_mem(IN NDIS_HANDLE MiniportAdapterHandle,
		    IN ULONG Length,
		    IN BOOLEAN Cached,
		    IN PVOID VirtualAddress,
		    IN NDIS_PHYSICAL_ADDRESS PhysicalAddress)
{
  NdisMFreeSharedMemory(MiniportAdapterHandle,
			Length,
			Cached,
			VirtualAddress,
			PhysicalAddress);
  if (VirtualAddress != NULL) {
    g_alloc_shared_mem_cnt--;
    g_alloc_shared_mem_len -= Length;
  }
}

void
mx__dbg_shared_mem(void)
{
  DbgPrint("g_alloc_shared_mem_cnt = %I64u\n", g_alloc_shared_mem_cnt);
  DbgPrint("g_alloc_shared_men_len = %I64u\n", g_alloc_shared_mem_len);
}

static unsigned _int64 g_alloc_tagged_mem_cnt = 0; /* number of allocations */
static unsigned _int64 g_alloc_tagged_mem_len = 0; /* total number of bytes */

NDIS_STATUS
mx__alloc_tagged_mem(OUT PVOID *VirtualAddress,
		     IN UINT Length,
		     IN ULONG Tag)
{
  NDIS_STATUS status;

  status = NdisAllocateMemoryWithTag(VirtualAddress,
				     Length,
				     Tag);
  if (status == NDIS_STATUS_SUCCESS) {
    g_alloc_tagged_mem_cnt++;
    g_alloc_tagged_mem_len += Length;
  }
  return status;
}

VOID
mx__free_mem(IN PVOID VirtualAddress,
	     IN UINT Length,
	     IN UINT MemoryFlags)
{
  NdisFreeMemory(VirtualAddress,
		 Length,
		 MemoryFlags);
  if (VirtualAddress != NULL) {
    g_alloc_tagged_mem_cnt--;
    g_alloc_tagged_mem_len -= Length;
  }
}

void
mx__dbg_tagged_mem(void)
{
  DbgPrint("g_alloc_tagged_mem_cnt = %I64u\n", g_alloc_tagged_mem_cnt);
  DbgPrint("g_alloc_tagged_mem_len = %I64u\n", g_alloc_tagged_mem_len);
}

/* mx_kfree doesn't have length as a parameter so g_kmem_len is only
 * useful to give an idea of how much memory is used. */
static unsigned _int64 g_kmem_cnt = 0; /* number of allocations */
static unsigned _int64 g_kmem_len = 0; /* total number of bytes */

void
mx__dbg_kmem(void)
{
  DbgPrint("g_kmem_cnt = %I64u\n", g_kmem_cnt);
  DbgPrint("g_kmem_len (estimated) = %I64u\n", g_kmem_len);
}


void
mx_fini_send(mx_adapter_t *adapter)
{
  int i;
  struct mx_ether *eth;
  void *va;
  NDIS_PHYSICAL_ADDRESS pa;

  eth = adapter->is->ether;
  for (i = 0; i < NUM_TX; ++i) {
    va = eth->tx.info[i].va;
    pa = eth->tx.info[i].pa;
    mx__free_shared_mem(adapter->miniport_adapter_handle,
			  MX_BIG_LEN, FALSE, va, pa);
  }
}

static void
mx_fini_recv(mx_adapter_t *adapter)
{
  int i;
  struct mx_ether *eth;
  void *va;
  NDIS_PHYSICAL_ADDRESS pa;

  DbgPrint("mx_fini_recv\n");

  eth = adapter->is->ether;
  for (i = 0; i < NUM_RX; ++i) {
    va = eth->rx_small.info[i].va;
    pa = eth->rx_small.info[i].pa;
    mx__free_shared_mem(adapter->miniport_adapter_handle,
			  MX_SMALL_LEN, FALSE, va, pa);
    /* TODO: assert that buffer len is as expected? */
    NdisFreeBuffer(eth->rx_small.info[i].buffer);
    NdisFreePacket(eth->rx_small.info[i].packet);
  }
  for (i = 0; i < NUM_RX; ++i) {
    va = eth->rx_big.info[i].va;
    pa = eth->rx_big.info[i].pa;
    mx__free_shared_mem(adapter->miniport_adapter_handle,
			  MX_BIG_LEN, FALSE, va, pa);
    /* TODO: assert that buffer len is as expected? */
    NdisFreeBuffer(eth->rx_big.info[i].buffer);
    NdisFreePacket(eth->rx_big.info[i].packet);
  }
}

static void
mx_free_buffers(mx_adapter_t *adapter)
{
  NdisFreeBufferPool(adapter->recv_large_buffer_pool);
  NdisFreePacketPool(adapter->recv_large_packet_pool);
  NdisFreeBufferPool(adapter->recv_small_buffer_pool);
  NdisFreePacketPool(adapter->recv_small_packet_pool);
}

static void
mx_ether_close(struct mx_ether *eth)
{
  mx_adapter_t *adapter;
  uint32_t dont_care;

  adapter = eth->arch.adapter;

  if (eth->running != MX_ETH_RUNNING) {
    return;
  }

  eth->running = MX_ETH_STOPPING;
  mx_lanai_command(eth->is, MX_MCP_CMD_ETHERNET_DOWN,
		   0, 0, 0, &dont_care, &eth->cmd_sync);
  eth->running = 0;

  /* Wait for packets given to upper layer to come back. */
  while ((eth->arch.next_small_to_fill + eth->arch.next_big_to_fill)
	 < (eth->rx_small.cnt + eth->rx_big.cnt)) {
    NdisMSleep(2000);
  }

  mx_fini_recv(adapter);
  mx_fini_send(adapter);
  mx_free_buffers(adapter);
  mx_ether_close_common(eth->is);
}

NTSTATUS
mx_stop_device(mx_instance_state_t *is)
{
  /* Tell the kwindow timer not to run again. Wait for acknowledgement. */
  is->arch.kwindow_timer_cancel = TRUE;
  NdisWaitEvent(&is->arch.kwindow_timer_event, 0);

  if (is->arch.intr_ready) {
    if (is->board_ops.disable_interrupt != NULL) {
      is->board_ops.disable_interrupt(is);
    }
    NdisMSleep(100);
    NdisMDeregisterInterrupt(&is->arch.miniport_interrupt);
    mx_instance_finalize(is);
  }

  return STATUS_SUCCESS;
}

void deregister_device(int idx);

VOID
mx_halt(IN NDIS_HANDLE MiniportAdapterContext)
{
  mx_adapter_t *adapter = (mx_adapter_t*)MiniportAdapterContext;
  mx_instance_state_t *is;
  struct mx_packetq_entry *entry;
  PNDIS_PACKET packet;
  int idx;

  MXE("---> mx_halt\n");
  /* Fail queued sends. */
  NdisAcquireSpinLock(&adapter->send_lock);
  while (!STAILQ_EMPTY(&adapter->sendq)) {
    entry = STAILQ_FIRST(&adapter->sendq);
    STAILQ_REMOVE_HEAD(&adapter->sendq, entries);
    packet = CONTAINING_RECORD(entry, NDIS_PACKET,
			       MiniportReserved[sizeof (void*)]);
    NdisMSendComplete(adapter->miniport_adapter_handle, packet,
		      NDIS_STATUS_FAILURE);
  }
  NdisReleaseSpinLock(&adapter->send_lock);
  /* Wait for sends given off to mcp to return. */
  while (adapter->is->ether->tx.done < adapter->is->ether->tx.req) {
    NdisMSleep(2000);
  }

#if MX_ENABLE_ETHERNET
  mx_ether_close(adapter->is->ether);
#endif
  mx_ether_detach(adapter->is);
  NdisMDeregisterAdapterShutdownHandler(adapter->miniport_adapter_handle);
  NdisFreeSpinLock(&adapter->recv_lock);
  NdisFreeSpinLock(&adapter->send_lock);
  is = adapter->is;

  mx_stop_device(is);
  /* Interrupts should be turned off at this point. */
  NdisMSleep(100);
  mx__free_mem(adapter, sizeof (*adapter), 0);
  mx_spin_lock_destroy(&is->arch.intr_lock);
  idx = is->arch.unit;
  deregister_device(idx);
  mx_unit_release(idx);
  mx_kfree(is);

  if (mx_num_instances == 0) {
    NdisMDeregisterDevice(g_mxctl_handle);
  }

  MXE("<--- mx_halt\n");
}

VOID
mx_shutdown(IN PVOID ShutdownContext)
{
  MXE("mx_shutdown\n");
}

void mx_ether_foo(void)
{
  DbgPrint("foo\n");
}

static int
mx_alloc_buffers(mx_adapter_t *adapter)
{
  NDIS_STATUS status;

  status = NdisMInitializeScatterGatherDma(adapter->miniport_adapter_handle,
					   TRUE, 9 * 1024); /* TODO: Magic */
  if (status != NDIS_STATUS_SUCCESS) {
    goto abort_with_nothing;
  }
  NdisAllocatePacketPoolEx(&status, &adapter->recv_small_packet_pool,
			   NUM_RX, 0xFFFF, PROTOCOL_RESERVED_SIZE_IN_PACKET);
  if (status != NDIS_STATUS_SUCCESS) {
    goto abort_with_nothing;
  }
  NdisAllocateBufferPool(&status, &adapter->recv_small_buffer_pool, NUM_RX);
  if (status != NDIS_STATUS_SUCCESS) {
    goto abort_with_recv_small_packet_pool;
  }
  NdisAllocatePacketPoolEx(&status, &adapter->recv_large_packet_pool,
			   NUM_RX, 0xFFFF, PROTOCOL_RESERVED_SIZE_IN_PACKET);
  if (status != NDIS_STATUS_SUCCESS) {
    goto abort_with_recv_small_buffer_pool;
  }
  NdisAllocateBufferPool(&status, &adapter->recv_large_buffer_pool, NUM_RX);
  if (status != NDIS_STATUS_SUCCESS) {
    goto abort_with_recv_large_packet_pool;
  }

  return 0;

 abort_with_recv_large_packet_pool:
  NdisFreePacketPool(adapter->recv_large_packet_pool);
 abort_with_recv_small_buffer_pool:
  NdisFreeBufferPool(adapter->recv_small_buffer_pool);
 abort_with_recv_small_packet_pool:
  NdisFreePacketPool(adapter->recv_small_packet_pool);
 abort_with_nothing:
  return 1;
}

static int
mx_init_send(mx_adapter_t *adapter)
{
  int i;
  struct mx_ether *eth;
  PVOID va;
  NDIS_PHYSICAL_ADDRESS pa;

  DbgPrint("mx_init_send\n");

  eth = adapter->is->ether;
  for (i = 0; i < NUM_TX; ++i) {
    mx__alloc_shared_mem(eth->arch.adapter->miniport_adapter_handle,
			      MX_BIG_LEN, FALSE, &va, &pa);
    if (va == NULL) {
      /* TODO: Error handling. */
      return 1;
    }
    eth->tx.info[i].va = va;
    eth->tx.info[i].pa = pa;
    eth->tx.info[i].packet = NULL;
  }
  return 0;
}

#define MX_PACKET_IDX(pkt_)\
(int*)(&(pkt_)->MiniportReserved[0])
#define MX_PACKET_Q(pkt_) \
(struct mx_packetq_entry*)(&(pkt_)->MiniportReserved[sizeof (int*)])
#define MX_PACKET_PA(pkt_) \
(PNDIS_PHYSICAL_ADDRESS)(&(pkt_)->MiniportReserved[sizeof (int*)])

int
mx_get_buf_small(struct mx_ether *eth, int idx)
{
  PVOID va;
  NDIS_PHYSICAL_ADDRESS pa;
  NDIS_STATUS status;
  PNDIS_PACKET packet;
  PNDIS_BUFFER buffer;

  mx__alloc_shared_mem(eth->arch.adapter->miniport_adapter_handle,
			    MX_SMALL_LEN, FALSE, &va, &pa);
  if (va == NULL) {
    return ENOMEM;
  }
  NdisAllocatePacket(&status, &packet,
		     eth->arch.adapter->recv_small_packet_pool);
  if (status != NDIS_STATUS_SUCCESS) {
    /* TODO */
  }
  NdisAllocateBuffer(&status, &buffer,
		     eth->arch.adapter->recv_small_buffer_pool,
		     (char*)va + MX_MCP_ETHER_PAD,
		     MX_SMALL_LEN - MX_MCP_ETHER_PAD);
  if (status != NDIS_STATUS_SUCCESS) {
    /* TODO */
  }
  NdisChainBufferAtFront(packet, buffer);
  NDIS_SET_PACKET_HEADER_SIZE(packet, 14); /* TODO: Magic number. */
  eth->rx_small.info[idx].va = va;
  eth->rx_small.info[idx].pa = pa;
  eth->rx_small.info[idx].packet = packet;
  eth->rx_small.info[idx].buffer = buffer;
  eth->rx_small.shadow[idx].addr_low = htonl(NdisGetPhysicalAddressLow(pa));
  eth->rx_small.shadow[idx].addr_high = htonl(NdisGetPhysicalAddressHigh(pa));

  /* copy 8 descriptors to the mcp at a time */
  if ((idx & 3) == 3) {
    /* FIXME: avoid 64 bytes write-combining */
    mx_pio_memcpy(&eth->rx_small.ring[idx - 3], &eth->rx_small.shadow[idx - 3],
		4 * sizeof (*eth->rx_small.ring), 0);
    MX_STBAR();
    MX_PIO_WRITE(eth->rx_small.lanai_cnt, htonl(eth->rx_small.cnt));
  }

  return 0;
}

int
mx_get_buf_big(struct mx_ether *eth, int idx)
{
  PVOID va;
  NDIS_PHYSICAL_ADDRESS pa;
  NDIS_STATUS status;
  PNDIS_PACKET packet;
  PNDIS_BUFFER buffer;

  mx__alloc_shared_mem(eth->arch.adapter->miniport_adapter_handle,
			    MX_BIG_LEN, FALSE, &va, &pa);
  if (va == NULL) {
    return ENOMEM;
  }
  NdisAllocatePacket(&status, &packet,
		     eth->arch.adapter->recv_large_packet_pool);
  if (status != NDIS_STATUS_SUCCESS) {
    /* TODO */
  }
  NdisAllocateBuffer(&status, &buffer,
		     eth->arch.adapter->recv_large_buffer_pool,
		     (char*)va + MX_MCP_ETHER_PAD,
		     MX_BIG_LEN - MX_MCP_ETHER_PAD);
  if (status != NDIS_STATUS_SUCCESS) {
    /* TODO */
  }
  NdisChainBufferAtFront(packet, buffer);
  NDIS_SET_PACKET_HEADER_SIZE(packet, 14); /* TODO: Magic number. */
  eth->rx_big.info[idx].va = va;
  eth->rx_big.info[idx].pa = pa;
  eth->rx_big.info[idx].packet = packet;
  eth->rx_big.info[idx].buffer = buffer;
  eth->rx_big.shadow[idx].addr_low = htonl(NdisGetPhysicalAddressLow(pa));
  eth->rx_big.shadow[idx].addr_high = htonl(NdisGetPhysicalAddressHigh(pa));

  /* copy 8 descriptors to the mcp at a time */
  if ((idx & 3) == 3) {
    /* FIXME: avoid 64 bytes write-combining */
    mx_pio_memcpy(&eth->rx_big.ring[idx - 3], &eth->rx_big.shadow[idx - 3],
		4 * sizeof (*eth->rx_big.ring), 0);
    MX_STBAR();
    MX_PIO_WRITE(eth->rx_big.lanai_cnt, htonl(eth->rx_big.cnt));
  }

  return 0;
}

static int
mx_init_recv(mx_adapter_t *adapter)
{
  int i;
  struct mx_ether *eth;

  eth = adapter->is->ether;

  for (i = 0; i < NUM_RX; ++i) {
    mx_get_buf_small(eth, i);
  }

  for (i = 0; i < NUM_RX; ++i) {
    mx_get_buf_big(eth, i);
  }

  return 0;
}

int
mx_ether_attach(mx_instance_state_t *is)
{
  NDIS_STATUS status;
  struct mx_ether *eth;
  int i;

  status = mx__alloc_tagged_mem(&eth, sizeof (*eth), 42);
  NdisZeroMemory(eth, sizeof (*eth));
  eth->is = is;
  is->ether = eth;
  for (i = 0; i < 6; ++i) {
    eth->current_mac[i] = is->mac_addr[i];
  }

  return 0;
}

void
mx_ether_detach(mx_instance_state_t *is)
{
  mx__free_mem(is->ether, sizeof (*is->ether), 0);
}

static int
mx_ether_open(struct mx_ether *eth)
{
  int error;
  uint32_t dont_care;
  mx_adapter_t *adapter;

  adapter = eth->arch.adapter;
  if (eth->running != MX_ETH_STOPPED) {
    return 0;
  }
  eth->running = MX_ETH_STARTING;
  DbgPrint("MX_WIN_MAX_ETHER_MTU %d\n", MX_WIN_MAX_ETHER_MTU);
  DbgPrint("MX_SMALL_LEN %d\n", MX_SMALL_LEN);
  DbgPrint("MX_BIG_POW2 %d\n", MX_BIG_POW2);
  error = mx_ether_open_common(eth->is, MX_WIN_MAX_ETHER_MTU, MX_SMALL_LEN,
			       MX_BIG_POW2);
  if (error) {
    DbgPrint("mx_ether_open_common failed!!!\n");
    goto abort_with_nothing;
  }

  if (mx_alloc_buffers(adapter) != 0) {
    error = 1;
    goto abort_with_ether_open_common;
  }
  if (mx_init_send(adapter) != 0) {
    DbgPrint("unable to allocate send buffers\n");
    goto abort_with_ether_open_common;
  }
  if (mx_init_recv(adapter) != 0) {
    DbgPrint("unable to allocate recv buffers\n");
    mx_fini_send(adapter);
    goto abort_with_ether_open_common;
  }

  /* recvs already allocated */
  eth->rx_small.cnt = NUM_RX;
  eth->arch.next_small_to_fill = NUM_RX;
  eth->rx_big.cnt = NUM_RX;
  eth->arch.next_big_to_fill = NUM_RX;
  MX_PIO_WRITE(eth->rx_small.lanai_cnt, htonl(eth->rx_small.cnt));
  MX_PIO_WRITE(eth->rx_big.lanai_cnt, htonl(eth->rx_big.cnt));

  error = mx_lanai_command(eth->is, MX_MCP_CMD_ETHERNET_UP, 0, 0, 0,
			   &dont_care, &eth->cmd_sync);
  mx_ether_start_common(eth->is, MX_WIN_MAX_ETHER_MTU, MX_SMALL_LEN,
			MX_BIG_POW2);
  eth->running = MX_ETH_RUNNING;
  return 0;

 abort_with_ether_open_common:
  mx_ether_close_common(eth->is);
 abort_with_nothing:
  eth->running = MX_ETH_STOPPED;
  return error;
}

int
mx_unit_alloc(void)
{
  int i;

  mx_spin_lock(&g_dev_lock);
  g_devices = mx_kmalloc(mx_max_instance * sizeof (*g_devices), MX_MZERO);
  g_instances = mx_kmalloc(mx_max_instance * sizeof (*g_instances), MX_MZERO);
  g_handles = mx_kmalloc(mx_max_instance * sizeof (*g_handles), MX_MZERO);
  if (g_devices != NULL && g_instances != NULL && g_handles != NULL) {
    for (i = 0; i < mx_max_instance; ++i) {
      g_devices[i] = NULL;
      g_instances[i] = NULL;
    }
    g_endpts = mx_kmalloc(mx_max_instance * mx_max_endpoints *
			 sizeof (*g_endpts), MX_MZERO);
    if (g_endpts != NULL) {
      for (i = 0; i < mx_max_instance * mx_max_endpoints; ++i) {
	g_endpts[i] = NULL;
      }
    }
    else {
      mx_kfree(g_devices);
      g_devices = NULL;
      mx_kfree(g_instances);
      g_instances = NULL;
      mx_kfree(g_handles);
    }
  }
  else {
    if (g_devices) {
      mx_kfree(g_devices);
      g_devices = NULL;
    }
    if (g_instances) {
      mx_kfree(g_instances);
      g_instances = NULL;
    }
    if (g_handles) {
      mx_kfree(g_handles);
    }
  }
  mx_spin_unlock(&g_dev_lock);
  return ((g_devices == NULL) && (g_instances != NULL) && (g_endpts == NULL));
}

void
mx_unit_free(void)
{
  mx_spin_lock(&g_dev_lock);
  mx_kfree(g_devices);
  mx_kfree(g_instances);
  mx_kfree(g_handles);
  mx_kfree(g_endpts);
  mx_spin_unlock(&g_dev_lock);
}

int
mx_unit_reserve(void)
{
  int i;

  ASSERT(g_devices != NULL);
  mx_spin_lock(&g_dev_lock);
  for (i = 0; i < mx_max_instance; ++i) {
    if (g_devices[i] == NULL) {
      g_devices[i] = (void*)-1;
      break;
    }
  }
  mx_spin_unlock(&g_dev_lock);
  return i;
}

/* -1 - mxctl
 * 0 - (mx_max_instance-1) - valid unit
 * mx_max_instance - invalid unit */
int
mx_unit_lookup(PDEVICE_OBJECT dev_obj)
{
  int i;

  ASSERT(g_devices != NULL);
  mx_spin_lock(&g_dev_lock);
  if (dev_obj == g_mxctl) {
    return -1;
  }
  for (i = 0; i < mx_max_instance; ++i) {
    if (g_devices[i] == dev_obj) {
      break;
    }
  }
  mx_spin_unlock(&g_dev_lock);
  return i;
}

/* is = NULL, es = NULL - mxctl
 * is != NULL, es = NULL - endptless
 * is != NULL, es != NULL - endpt */
void
mx_decode_context(PFILE_OBJECT file_obj, mx_instance_state_t **is,
		  mx_endpt_state_t **es, int *privileged)
{
  mx_assert(file_obj->FsContext != NULL);
  *is = (mx_instance_state_t*) file_obj->FsContext;
  *es = (mx_endpt_state_t*) file_obj->FsContext2;
  if (file_obj->FileName.Length == 0) {
    /* mxctl or mx%d */
    *privileged = 0;
  }
  else if ((wcscmp(file_obj->FileName.Buffer, L"\\p") == 0) ||
	   (wcscmp(file_obj->FileName.Buffer, L"\\r") == 0)) {
    /* mxctl\p, mx%d\p, or mx%d\r */
    *privileged = 1;
  }
  else {
    /* mx%de%d */
    *privileged = 0;
  }
}

void
mx_munmapall(mx_endpt_state_t *es)
{
  struct mmapmd_entry *entry;

  DbgPrint("mx_munmapall\n");

  while (!SLIST_EMPTY(&es->arch.mmapmd)) {
    entry = SLIST_FIRST(&es->arch.mmapmd);
    SLIST_REMOVE_HEAD(&es->arch.mmapmd, entries);
    MmUnmapLockedPages(entry->uva, entry->mdl);
    IoFreeMdl(entry->mdl);
    mx_kfree(entry);
  }
}

int
mx_mmap(mx_endpt_state_t *es, mx_uaddr_t arg)
{
  mx_mmap_t i;
  int status;
  void *kva, *va;
  struct mmapmd_entry *entry;
  mx_page_pin_t *dontcare;
  unsigned long off, pos, len;
  PMDL mdl;
  int mem_type;

  DbgPrint("mx_mmap\n");

  status = mx_copyin(arg, &i, sizeof(i), es->is_kernel);
  if (status)
    goto abort_with_nothing;

  off = i.offset;
  len = i.len;
  /* validate mapping */

  if (off & (PAGE_SIZE - 1)) {
    status = ENXIO;
    goto abort_with_nothing;
  }

  mx_mutex_enter(&es->sync);
  for (pos = 0; pos < len; pos += PAGE_SIZE) {
    /*
     * See if each page of the request is in range
     */
    
    status = mx_mmap_off_to_kva(es, off + pos, &kva, &mem_type, &dontcare);
    if (status != 0) {
      DbgPrint("status =%d, pos = %x%lx, len = %x%lx\n", status, pos, len);
      goto abort_with_mutex;
    }
  }
  

  /* map -- this routine returns a kernel virtual address */
  mx_mmap_off_to_kva(es, off, &kva, &mem_type, &dontcare);  /* can't fail -- it worked above! */

  entry = (struct mmapmd_entry *) mx_kmalloc(sizeof (*entry), MX_MZERO|MX_WAITOK);
  if (entry == NULL) {
    MX_WARN(("mx_kmalloc failed for md list entry\n"));
    status = ENXIO;
    goto abort_with_mutex;
  }	
  mdl = IoAllocateMdl(kva, len, FALSE, FALSE, NULL);
  if (mdl == NULL) {
    status = ENXIO;
    goto abort_with_entry;
  }
  MmBuildMdlForNonPagedPool(mdl);
  __try {
    if (mem_type == MX_MEM_HOSTMEM) {
      va = MmMapLockedPagesSpecifyCache(mdl, UserMode, MmCached, NULL,
					FALSE, NormalPagePriority);
    }
    else if (mem_type == MX_MEM_SRAM) {
      va = MmMapLockedPagesSpecifyCache(mdl, UserMode, MmWriteCombined, NULL,
					FALSE, NormalPagePriority);
    }
    else {
      va = NULL;
      MX_WARN(("unexpected mem_type %d\n", mem_type));
    }
  }
  __except (EXCEPTION_EXECUTE_HANDLER) {
    MX_WARN(("MmMapLockedPagesSpecifyCache failed with 0x%x\n",
	     GetExceptionCode()));
    va = NULL;
  }
  if (va == NULL) {
    status = ENXIO;
    goto abort_with_mdl;
  }
  i.va = (uint64_t)(uintptr_t)va;

  /* Tell the user where to find it */
  status = mx_copyout (&i, arg, sizeof(i), es->is_kernel);
  if (status) {	
	  MX_WARN(("mx_mmap: copyout failed -  addr 0x%"PRIx64"\n", 
		  (uint64_t)arg));
    goto abort_with_va;
  }

  /* save the map entry on the arch-specific endpt "entry" list.  It
   * will be freed when the endpt is closed or when mx_munmap is called.
   */
  entry->uva = va;
  entry->mdl = mdl;
  SLIST_INSERT_HEAD(&es->arch.mmapmd, entry, entries);

  mx_mutex_exit(&es->sync);

  return 0;


 abort_with_va:
  MmUnmapLockedPages(va, mdl);

 abort_with_mdl:
  IoFreeMdl(mdl);

 abort_with_entry:
  mx_kfree(entry);

 abort_with_mutex:
  mx_mutex_exit(&es->sync);

 abort_with_nothing:
  return status;
}

static int
mx_set_endpoint(mx_endpt_state_t *es, int unit, int privileged,
		int endpoint, mx_uaddr_t addr, int raw)
{
  mx_set_endpt_t set_endpt;
  int status;

  if (!raw) {
    status = mx_arch_copyin(addr, &set_endpt, sizeof (set_endpt));
    if (status) {
      /* TODO: Use an errno value? */
      return status;
    }

    if (set_endpt.endpoint != endpoint) {
      return ENXIO;
    }
  }

  es->opener.pid = 0; /* TODO: Get pid. */
  es->privileged = privileged;
  es->is_kernel = 0;
  status = mx_common_open(unit, set_endpt.endpoint, es, raw);
  set_endpt.session_id = es->session_id;

  if (status != 0) {
    es->is = 0;
  }

  if (!raw) {
    status |= mx_arch_copyout(&set_endpt, addr, sizeof (set_endpt));
  }
  return status;
}

void mx_hang_test(int i)
{
#if 0
  KWAIT_REASON reason;
  KPROCESSOR_MODE mode;
  BOOLEAN alertable;
  NTSTATUS status;
  LARGE_INTEGER timeout;
  
  /*
   * A mode of UserMode means the application can ^C out. In this
   * situation, the value of alertable doesn't matter because the ^C
   * puts the thread in an alertable state. If you ^C, the status
   * returned is STATUS_USER_APC.
   */

  switch (i) {
  case 1: reason = Executive; mode = KernelMode; alertable = TRUE; break;
  case 2: reason = UserRequest; mode = KernelMode; alertable = TRUE; break;
  case 3: reason = Executive; mode = UserMode; alertable = TRUE; break;
  case 4: reason = UserRequest; mode = UserMode; alertable = TRUE; break;
  case 5: reason = Executive; mode = KernelMode; alertable = FALSE; break;
  case 6: reason = UserRequest; mode = KernelMode; alertable = FALSE; break;
  case 7: reason = Executive; mode = UserMode; alertable = FALSE; break;
  case 8: reason = UserRequest; mode = UserMode; alertable = FALSE; break;
  default:
    DbgPrint("unknown hang %d\n", i);
    return;
  }
  if (reason == Executive) {
    DbgPrint("Executive:");
  }
  else {
    DbgPrint("UserRequest:");
  }
  if (mode == KernelMode) {
    DbgPrint("KernelMode:");
  }
  else {
    DbgPrint("UserMode:");
  }
  if (alertable == TRUE) {
    DbgPrint("TRUE\n");
  }
  else {
    DbgPrint("FALSE\n");
  }

  timeout.QuadPart = 10; /* seconds */
  timeout.QuadPart *= 1000; /* milliseconds */
  timeout.QuadPart *= 1000; /* microseconds */
  timeout.QuadPart *= 10; /* 100 nanosecond intervals */
  timeout.QuadPart *= -1; /* relative time */
  
  status = KeWaitForSingleObject(&g_event, reason, mode, alertable,
				 &timeout);
  switch(status) {
  case STATUS_SUCCESS:
    DbgPrint("wait success\n");
    break;
  case STATUS_ALERTED:
    DbgPrint("wait alerted\n");
    break;
  case STATUS_USER_APC:
    DbgPrint("wait user apc\n");
    break;
  case STATUS_TIMEOUT:
    DbgPrint("wait timeout\n");
    break;
  }
#endif
}

NDIS_STATUS
get_translated_resources (mx_instance_state_t *is)
{
  CM_RESOURCE_LIST res[8];
  PCM_RESOURCE_LIST resources;
  PCM_FULL_RESOURCE_DESCRIPTOR full_descriptor;
  PCM_PARTIAL_RESOURCE_LIST partial_list;
  PCM_PARTIAL_RESOURCE_DESCRIPTOR partial_descriptor;
  ULONG ul;
  BOOLEAN have_interrupt = FALSE;
  BOOLEAN have_memory = FALSE;
  uint16_t vendor;
  uint16_t device;

  DbgPrint("get_translated_resources\n");

  if (mx_read_pci_config_16(is, offsetof(mx_pci_config_t, Vendor_ID),
			    &vendor)) {
    DbgPrint("Could not determine board vendor id\n");
    return NDIS_STATUS_FAILURE;
  }
  if (mx_read_pci_config_16(is, offsetof(mx_pci_config_t, Device_ID),
			    &device)) {
    DbgPrint("Could not determine board device id\n");
    return NDIS_STATUS_FAILURE;
  }

  resources = res;
  NdisMGetDeviceProperty (is->arch.miniportAdapterHandle, NULL, NULL, NULL,
			  NULL, &resources);
  full_descriptor = resources->List;
  partial_list = &full_descriptor->PartialResourceList;
  for (ul = 0; ul < partial_list->Count; ++ul)
    {
      partial_descriptor = &partial_list->PartialDescriptors[ul];
      switch (partial_descriptor->Type)
	{
	case CmResourceTypeInterrupt:
	  is->arch.level = (KIRQL) partial_descriptor->u.Interrupt.Level;
	  is->arch.vector =
	    partial_descriptor->u.Interrupt.Vector;
	  is->arch.affinity = partial_descriptor->u.Interrupt.Affinity;
	  DbgPrint("interrupt\n");
	  DbgPrint("level = %d\n", (int)is->arch.level);
	  DbgPrint("vector = %d\n", (int)is->arch.vector);
	  DbgPrint("affinity = %d\n", (int)is->arch.affinity);
	  have_interrupt = TRUE;
	  break;
	case CmResourceTypeMemory:
	  /* Myri10G cards have two memory regions. The 1M region is for
	     the special registers. */
	  if (vendor == 0x14c1 && device == 0x0008 &&
	      partial_descriptor->u.Memory.Length == 0x00100000) {
	    is->arch.special_base = partial_descriptor->u.Memory.Start;
	    is->arch.special_span = partial_descriptor->u.Memory.Length;
	    break;
	  }

	  is->arch.iomem_base =
	    partial_descriptor->u.Memory.Start;
	  is->board_span = partial_descriptor->u.Memory.Length;
	  DbgPrint("memory\n");
	  DbgPrint("base.high = %d\n", (int)is->arch.iomem_base.HighPart);
	  DbgPrint("base.low = %d\n", (int)is->arch.iomem_base.LowPart);
	  DbgPrint("span = %d\n", (int)is->board_span);
	  have_memory = TRUE;
	  break;
	}
    }
  if (!have_interrupt || !have_memory) {
    DbgPrint("have_interrupt = %d\n", (int)have_interrupt);
    DbgPrint("have_memory = %d\n", (int)have_memory);
    return NDIS_STATUS_FAILURE;
  }
  return NDIS_STATUS_SUCCESS;
}

void query_adapter_resources(mx_instance_state_t *is, NDIS_HANDLE wcc)
{
  NDIS_STATUS status;
  NDIS_RESOURCE_LIST reslist[4];
  PNDIS_RESOURCE_LIST resList;
  UINT bufsize;
  ULONG index;
  PCM_PARTIAL_RESOURCE_DESCRIPTOR pResDesc;
  uint16_t vendor;
  uint16_t device;

  DbgPrint("query_adapter_resources\n");

  mx_read_pci_config_16(is, offsetof(mx_pci_config_t, Vendor_ID),
			&vendor);
  mx_read_pci_config_16(is, offsetof(mx_pci_config_t, Device_ID),
			&device);

  //DbgBreakPoint();

  resList = reslist;
  bufsize = sizeof(reslist);
  NdisMQueryAdapterResources(&status, wcc, resList, &bufsize);

  for (index=0; index < resList->Count; index++)
    {
      pResDesc = &resList->PartialDescriptors[index];
      
      switch(pResDesc->Type)
	{
	case CmResourceTypePort:
	  DbgPrint("port\n");
	  break;
	  
	case CmResourceTypeInterrupt:
	  is->arch.level = (KIRQL)pResDesc->u.Interrupt.Level;
	  is->arch.vector = pResDesc->u.Interrupt.Vector;
	  is->arch.affinity = pResDesc->u.Interrupt.Affinity;
	  DbgPrint("interrupt\n");
	  DbgPrint("level = %d\n", (int)is->arch.level);
	  DbgPrint("vector = %d\n", (int)is->arch.vector);
	  DbgPrint("affinity = %d\n", (int)is->arch.affinity);
	  break;
	  
	case CmResourceTypeMemory:
	  /* Myri10G cards have two memory regions. The 1M region is for
	     the special registers. */
	  if (vendor == 0x14c1 && device == 0x0008 &&
	      pResDesc->u.Memory.Length == 0x00100000) {
	    is->arch.special_base = pResDesc->u.Memory.Start;
	    is->arch.special_span = pResDesc->u.Memory.Length;
	    break;
	  }

	  is->arch.iomem_base = pResDesc->u.Memory.Start;
	  is->board_span = pResDesc->u.Memory.Length;
	  DbgPrint("memory\n");
	  DbgPrint("base.high = %d\n", (int)is->arch.iomem_base.HighPart);
	  DbgPrint("base.low = %d\n", (int)is->arch.iomem_base.LowPart);
	  DbgPrint("span = %d\n", (int)is->board_span);
	  break;
	}
    }
}

NTSTATUS mx_mj_create(IN PDEVICE_OBJECT DeviceObject, IN PIRP Irp);
NTSTATUS mx_mj_cleanup(IN PDEVICE_OBJECT DeviceObject, IN PIRP Irp);
NTSTATUS mx_mj_close(IN PDEVICE_OBJECT DeviceObject, IN PIRP Irp);
NTSTATUS mx_mj_device_control(IN PDEVICE_OBJECT DeviceObject, IN PIRP Irp);

NDIS_STATUS
register_device(int idx, mx_instance_state_t *is)
{
  NDIS_STATUS status;
  NDIS_STRING dev_name;
  NDIS_STRING sym_name;
  wchar_t dev_buff[80];
  wchar_t sym_buff[80];
  PDRIVER_DISPATCH mj_func[IRP_MJ_MAXIMUM_FUNCTION+1];
  PDEVICE_OBJECT pdevobj;
  NDIS_HANDLE devhandle;

  swprintf(dev_buff, L"\\Device\\mx%d", idx);
  RtlInitUnicodeString(&dev_name, dev_buff);
  swprintf(sym_buff, L"\\DosDevices\\mx%d", idx);
  RtlInitUnicodeString(&sym_name, sym_buff);
  NdisZeroMemory(mj_func, (IRP_MJ_MAXIMUM_FUNCTION+1) * sizeof (PDRIVER_DISPATCH));
  mj_func[IRP_MJ_CREATE] = mx_mj_create;
  mj_func[IRP_MJ_CLEANUP] = mx_mj_cleanup;
  mj_func[IRP_MJ_CLOSE] = mx_mj_close;
  mj_func[IRP_MJ_DEVICE_CONTROL] = mx_mj_device_control;
  status = NdisMRegisterDevice(g_ndis_wrapper, &dev_name, &sym_name,
			       mj_func, &pdevobj, &devhandle);
  if (status == NDIS_STATUS_SUCCESS) {
    g_devices[idx] = pdevobj;
    g_handles[idx] = devhandle;
    g_instances[idx] = is;
  }

  return status;
}

void
deregister_device(int idx)
{
  NdisMDeregisterDevice(g_handles[idx]);
}

void mx_windows_kwindow_timer(IN PVOID SystemSpecific1,
			      IN PVOID FunctionContext,
			      IN PVOID SystemSpecific2,
			      IN PVOID SystemSpecific3)
{
  mx_instance_state_t * is = FunctionContext;
  LARGE_INTEGER li;

  KeQueryTickCount(&li);
  is->kernel_window->jiffies = li.LowPart;

  if (!is->arch.kwindow_timer_cancel) {
    NdisSetTimer(&is->arch.kwindow_timer, 10);
  }
  else {
    NdisSetEvent(&is->arch.kwindow_timer_event);
  }
}

NDIS_STATUS
mx_initialize(OUT PNDIS_STATUS OpenErrorStatus,
	      OUT PUINT SelectedMediumIndex,
	      IN PNDIS_MEDIUM MediumArray,
	      IN UINT MediumArraySize,
	      IN NDIS_HANDLE MiniportAdapterHandle,
	      IN NDIS_HANDLE WrapperConfigurationContext)
{
  NDIS_STATUS status = NDIS_STATUS_SUCCESS;
  UINT i;
  mx_adapter_t *adapter;
  mx_instance_state_t *is;
  int idx;
  ULONG property_addr;
  ULONG ul;

  MXE("---> mx_initialize\n");
  //DbgBreakPoint();

  for (i = 0; i < MediumArraySize; ++i) {
    if (MediumArray[i] == NdisMedium802_3) {
      *SelectedMediumIndex = i;
      break;
    }
  }
  if (i == MediumArraySize) {
    status = NDIS_STATUS_UNSUPPORTED_MEDIA;
    MXE("unsupported media\n");
    goto abort_with_nothing;
  }

  idx = mx_unit_reserve();

  is = mx_kmalloc(sizeof (*is), MX_MZERO);
  is->arch.unit = idx;
  is->arch.miniportAdapterHandle = MiniportAdapterHandle;
  /* TODO: NdisMRegisterDevice */
  get_translated_resources(is);
  query_adapter_resources(is, WrapperConfigurationContext);

  status = mx__alloc_tagged_mem(&adapter, sizeof (*adapter), MX_TAG);
  if (status != NDIS_STATUS_SUCCESS) {
    status = NDIS_STATUS_RESOURCES;
    MXE("unable to allocate instance\n");
    goto abort_with_nothing;
  }
  NdisZeroMemory(adapter, sizeof (*adapter));

  /* We may possibly need to tweak our pci bridge. */
  NdisMGetDeviceProperty(MiniportAdapterHandle, &adapter->pdo, NULL,
			 NULL, NULL, NULL); /* Get pdo. */
  IoGetDeviceProperty(adapter->pdo, DevicePropertyAddress, sizeof (ULONG),
		      &property_addr, &ul);
  adapter->func_num = property_addr & 0x0000ffff;
  adapter->dev_num = (property_addr >> 16) & 0x0000ffff;
  IoGetDeviceProperty(adapter->pdo, DevicePropertyBusNumber,
		      sizeof (ULONG), &adapter->bus_num, &ul);
  DbgPrint("func = %lu\n", adapter->func_num);
  DbgPrint("dev = %lu\n", adapter->dev_num);
  DbgPrint("bus = %lu\n", adapter->bus_num);
  mx_frob_chipsets(adapter->bus_num);
  
  adapter->is = is;
  is->arch.adapter = adapter;
  adapter->miniport_adapter_handle = MiniportAdapterHandle;
  adapter->filter = MX_PACKET_FILTER;
  adapter->lookahead = MX_LOOKAHEAD;
  NdisAllocateSpinLock(&adapter->send_lock);
  NdisAllocateSpinLock(&adapter->recv_lock);
  STAILQ_INIT(&adapter->sendq);

  NdisMSetAttributesEx(MiniportAdapterHandle, adapter, 0,
		       (NDIS_ATTRIBUTE_BUS_MASTER |
			NDIS_ATTRIBUTE_DESERIALIZE |
			NDIS_ATTRIBUTE_USES_SAFE_BUFFER_APIS),
		       NdisInterfacePci);

  status = register_device(idx, is);

  NdisMRegisterAdapterShutdownHandler(MiniportAdapterHandle, adapter,
				      mx_shutdown);

  mx_ether_attach(adapter->is);
  adapter->is->ether->arch.adapter = adapter;

  //DbgBreakPoint();
  mx_spin_lock_init(&is->arch.intr_lock, NULL, 0, NULL);
  is->arch.intr_ready = 0;
#if 0
  status = NdisMRegisterInterrupt(&is->arch.miniport_interrupt,
				  MiniportAdapterHandle,
				  is->arch.vector, is->arch.level, TRUE,
				  TRUE, NdisInterruptLevelSensitive);
#else
  status = NdisMRegisterInterrupt(&is->arch.miniport_interrupt,
				  MiniportAdapterHandle,
				  is->arch.level, is->arch.level, TRUE,
				  TRUE, NdisInterruptLevelSensitive);
#endif
  is->arch.intr_ready = 1;

  is->kernel_window = mx_kmalloc(PAGE_SIZE, MX_MZERO);
  if (!is->kernel_window) {
    MX_NOTE (("Failed to allocate kernel window\n"));
    status = NDIS_STATUS_RESOURCES;
    goto abort_with_interrupt;
  }
  /* QueryIncrement is in 100ns unit => HZ=1e7/val */
  is->kernel_window->hz = 10000000 / KeQueryTimeIncrement();

  if (mx_instance_init(is, is->arch.unit) != 0) {
    DbgPrint("mx_instance_init failed!!!\n");
    status = NDIS_STATUS_RESOURCES;
    goto abort_with_kernel_window;
  }

  is->arch.kwindow_timer_cancel = FALSE;
  NdisInitializeEvent(&is->arch.kwindow_timer_event);
  NdisInitializeTimer(&is->arch.kwindow_timer, mx_windows_kwindow_timer,
		      is);
  NdisSetTimer(&is->arch.kwindow_timer, 10);

  mx_mutex_exit(&is->sync);

#if MX_ENABLE_ETHERNET
  if (mx_ether_open(adapter->is->ether) != 0) {
    status = NDIS_STATUS_RESOURCES;
    goto abort_with_ether_attach;
  }
#endif

  MXE("<--- mx_initialize\n");
  return status;

 abort_with_ether_attach:
  mx_ether_detach(adapter->is);
  NdisMDeregisterAdapterShutdownHandler(MiniportAdapterHandle);
  NdisFreeSpinLock(&adapter->recv_lock);
  NdisFreeSpinLock(&adapter->send_lock);
  mx__free_mem(adapter, sizeof (*adapter), 0);
 abort_with_kernel_window:
  mx_kfree(is->kernel_window);
 abort_with_interrupt:
  NdisMDeregisterInterrupt(&is->arch.miniport_interrupt);
 abort_with_nothing:
  return status;
}

/*{{{ query information */
#define MX_HEADER_SIZE 14
#if MX_MUST_DO_4K_ALIGN
#define MX_DATA_SIZE 1500
#define MX_HW_CHECKSUM 0
#else
#define MX_DATA_SIZE 9000
#define MX_HW_CHECKSUM 1
#endif

#if MX_HW_CHECKSUM
NDIS_TASK_TCP_IP_CHECKSUM g_task_tcp_ip_checksum = {
  {1, 1, 1, 1, 0},
  {1, 1, 1, 1, 0},
  {0, 0, 0, 0},
  {0, 0, 0, 0}
};
#else
NDIS_TASK_TCP_IP_CHECKSUM g_task_tcp_ip_checksum = {
  {0, 0, 0, 0, 0},
  {0, 0, 0, 0, 0},
  {0, 0, 0, 0},
  {0, 0, 0, 0}
};
#endif

NDIS_TASK_OFFLOAD g_task_offload = {
  NDIS_TASK_OFFLOAD_VERSION,
  sizeof (NDIS_TASK_OFFLOAD),
  TcpIpChecksumNdisTask,
  0,
  sizeof (NDIS_TASK_TCP_IP_CHECKSUM)
};

NDIS_STATUS 
mx_query_information(IN NDIS_HANDLE MiniportAdapterContext,
		     IN NDIS_OID Oid,
		     IN PVOID InformationBuffer,
		     IN ULONG InformationBufferLength,
		     OUT PULONG BytesWritten,
		     OUT PULONG BytesNeeded)
{
  mx_adapter_t *adapter = (mx_adapter_t*)MiniportAdapterContext;
  ULONG ul;
  USHORT us;
  NDIS_HARDWARE_STATUS hw_status = NdisHardwareStatusReady;
  NDIS_MEDIUM medium = NdisMedium802_3;
  char *ven_desc = "myvmini";
  char macaddr[6];
  ULONG length;
  PNDIS_TASK_OFFLOAD_HEADER task_offload_header;
  PNDIS_TASK_OFFLOAD task_offload;
  PNDIS_TASK_TCP_IP_CHECKSUM tcp_ip_checksum;

  NdisMoveMemory(macaddr, adapter->is->mac_addr, 6);

  switch (Oid) {

  case OID_GEN_SUPPORTED_LIST:
    if (InformationBufferLength < sizeof (g_oids)) {
      *BytesWritten = 0;
      *BytesNeeded = sizeof (g_oids);
      return NDIS_STATUS_INVALID_LENGTH;
    }
    NdisMoveMemory(InformationBuffer, g_oids, sizeof (g_oids));
    *BytesWritten = sizeof (g_oids);
    return NDIS_STATUS_SUCCESS;

  case OID_GEN_HARDWARE_STATUS:
    if (InformationBufferLength < sizeof (NDIS_HARDWARE_STATUS)) {
      *BytesWritten = 0;
      *BytesNeeded = sizeof (NDIS_HARDWARE_STATUS);
      return NDIS_STATUS_INVALID_LENGTH;
    }
    NdisMoveMemory(InformationBuffer, &hw_status, sizeof (hw_status));
    *BytesWritten = sizeof (hw_status);
    return NDIS_STATUS_SUCCESS;

  case OID_GEN_MEDIA_SUPPORTED:
    if (InformationBufferLength < sizeof (medium)) {
      *BytesWritten = 0;
      *BytesNeeded = sizeof (medium);
      return NDIS_STATUS_INVALID_LENGTH;
    }
    NdisMoveMemory(InformationBuffer, &medium, sizeof (medium));
    *BytesWritten = sizeof (medium);
    return NDIS_STATUS_SUCCESS;

  case OID_GEN_MEDIA_IN_USE:
    if (InformationBufferLength < sizeof (medium)) {
      *BytesWritten = 0;
      *BytesNeeded = sizeof (medium);
      return NDIS_STATUS_INVALID_LENGTH;
    }
    NdisMoveMemory(InformationBuffer, &medium, sizeof (medium));
    *BytesWritten = sizeof (medium);
    return NDIS_STATUS_SUCCESS;

  case OID_GEN_MAXIMUM_LOOKAHEAD:
    if (InformationBufferLength < sizeof (ULONG)) {
      *BytesWritten = 0;
      *BytesNeeded = sizeof (ULONG);
      return NDIS_STATUS_INVALID_LENGTH;
    }
    ul = MX_DATA_SIZE;
    NdisMoveMemory(InformationBuffer, &ul, sizeof (ul));
    *BytesWritten = sizeof (ul);
    return NDIS_STATUS_SUCCESS;

  case OID_GEN_MAXIMUM_FRAME_SIZE:
    if (InformationBufferLength < sizeof (ULONG)) {
      *BytesWritten = 0;
      *BytesNeeded = sizeof (ULONG);
      return NDIS_STATUS_INVALID_LENGTH;
    }
    ul = MX_DATA_SIZE;
    NdisMoveMemory(InformationBuffer, &ul, sizeof (ul));
    *BytesWritten = sizeof (ul);
    return NDIS_STATUS_SUCCESS;
    
  case OID_GEN_LINK_SPEED:
    if (InformationBufferLength < sizeof (ULONG)) {
      *BytesWritten = 0;
      *BytesNeeded = sizeof (ULONG);
      return NDIS_STATUS_INVALID_LENGTH;
    }
    switch (adapter->is->board_type) {
    case MX_BOARD_TYPE_D:
      ul = 20000000;
      break;
    case MX_BOARD_TYPE_E:
      ul = 40000000;
      break;
    case MX_BOARD_TYPE_Z:
      ul = 100000000;
      break;
    default:
      DbgPrint("unknown board type %d\n", (int)adapter->is->board_type);
      ul = 0;
      break;
    }
    NdisMoveMemory(InformationBuffer, &ul, sizeof (ul));
    *BytesWritten = sizeof (ul);
    return NDIS_STATUS_SUCCESS;

  case OID_GEN_TRANSMIT_BUFFER_SPACE:
    if (InformationBufferLength < sizeof (ULONG)) {
      *BytesWritten = 0;
      *BytesNeeded = sizeof (ULONG);
      return NDIS_STATUS_INVALID_LENGTH;
    }
    ul = (MX_HEADER_SIZE + MX_DATA_SIZE) * NUM_TX / 2;
    NdisMoveMemory(InformationBuffer, &ul, sizeof (ul));
    *BytesWritten = sizeof (ul);
    return NDIS_STATUS_SUCCESS;

  case OID_GEN_RECEIVE_BUFFER_SPACE:
    if (InformationBufferLength < sizeof (ULONG)) {
      *BytesWritten = 0;
      *BytesNeeded = sizeof (ULONG);
      return NDIS_STATUS_INVALID_LENGTH;
    }
    ul = (MX_HEADER_SIZE + MX_DATA_SIZE) * NUM_RX;
    NdisMoveMemory(InformationBuffer, &ul, sizeof (ul));
    *BytesWritten = sizeof (ul);
    return NDIS_STATUS_SUCCESS;

  case OID_GEN_TRANSMIT_BLOCK_SIZE:
    if (InformationBufferLength < sizeof (ULONG)) {
      *BytesWritten = 0;
      *BytesNeeded = sizeof (ULONG);
      return NDIS_STATUS_INVALID_LENGTH;
    }
    ul = (MX_HEADER_SIZE + MX_DATA_SIZE);
    NdisMoveMemory(InformationBuffer, &ul, sizeof (ul));
    *BytesWritten = sizeof (ul);
    return NDIS_STATUS_SUCCESS;

  case OID_GEN_RECEIVE_BLOCK_SIZE:
    if (InformationBufferLength < sizeof (ULONG)) {
      *BytesWritten = 0;
      *BytesNeeded = sizeof (ULONG);
      return NDIS_STATUS_INVALID_LENGTH;
    }
    ul = (MX_HEADER_SIZE + MX_DATA_SIZE);
    NdisMoveMemory(InformationBuffer, &ul, sizeof (ul));
    *BytesWritten = sizeof (ul);
    return NDIS_STATUS_SUCCESS;

  case OID_GEN_VENDOR_ID:
    if (InformationBufferLength < sizeof (ULONG)) {
      *BytesWritten = 0;
      *BytesNeeded = sizeof (ULONG);
      return NDIS_STATUS_INVALID_LENGTH;
    }
    ul = 0x0060dd00;
    NdisMoveMemory(InformationBuffer, &ul, sizeof (ul));
    *BytesWritten = sizeof (ul);
    return NDIS_STATUS_SUCCESS;

  case OID_GEN_VENDOR_DESCRIPTION:
    if (InformationBufferLength < (strlen(ven_desc)+1)) {
      *BytesWritten = 0;
      *BytesNeeded = strlen(ven_desc)+1;
      return NDIS_STATUS_INVALID_LENGTH;
    }
    NdisMoveMemory(InformationBuffer, ven_desc, strlen(ven_desc)+1);
    *BytesWritten = strlen(ven_desc)+1;
    return NDIS_STATUS_SUCCESS;

  case OID_GEN_VENDOR_DRIVER_VERSION:
    if (InformationBufferLength < sizeof (ULONG)) {
      *BytesWritten = 0;
      *BytesNeeded = sizeof (ULONG);
      return NDIS_STATUS_INVALID_LENGTH;
    }
    ul = 0x00010000;
    NdisMoveMemory(InformationBuffer, &ul, sizeof (ul));
    *BytesWritten = sizeof (ul);
    return NDIS_STATUS_SUCCESS;    

  case OID_GEN_CURRENT_PACKET_FILTER:
    if (InformationBufferLength < sizeof (ULONG)) {
      *BytesWritten = 0;
      *BytesNeeded = sizeof (ULONG);
      return NDIS_STATUS_INVALID_LENGTH;
    }
    ul = adapter->filter;
    NdisMoveMemory(InformationBuffer, &ul, sizeof (ul));
    *BytesWritten = sizeof (ul);
    return NDIS_STATUS_SUCCESS;    

  case OID_GEN_CURRENT_LOOKAHEAD:
    if (InformationBufferLength < sizeof (ULONG)) {
      *BytesWritten = 0;
      *BytesNeeded = sizeof (ULONG);
      return NDIS_STATUS_INVALID_LENGTH;
    }
    ul = MX_DATA_SIZE;
    NdisMoveMemory(InformationBuffer, &ul, sizeof (ul));
    *BytesWritten = sizeof (ul);
    return NDIS_STATUS_SUCCESS;    

  case OID_GEN_DRIVER_VERSION:
    if (InformationBufferLength < sizeof (USHORT)) {
      *BytesWritten = 0;
      *BytesNeeded = sizeof (USHORT);
      return NDIS_STATUS_INVALID_LENGTH;
    }
    us = 0x0500;
    NdisMoveMemory(InformationBuffer, &us, sizeof (us));
    *BytesWritten = sizeof (us);
    return NDIS_STATUS_SUCCESS;    

  case OID_GEN_MAXIMUM_TOTAL_SIZE:
    if (InformationBufferLength < sizeof (ULONG)) {
      *BytesWritten = 0;
      *BytesNeeded = sizeof (ULONG);
      return NDIS_STATUS_INVALID_LENGTH;
    }
    ul = (MX_HEADER_SIZE + MX_DATA_SIZE);
    NdisMoveMemory(InformationBuffer, &ul, sizeof (ul));
    *BytesWritten = sizeof (ul);
    return NDIS_STATUS_SUCCESS;    

  case OID_GEN_MAC_OPTIONS:
    if (InformationBufferLength < sizeof (ULONG)) {
      *BytesWritten = 0;
      *BytesNeeded = sizeof (ULONG);
      return NDIS_STATUS_INVALID_LENGTH;
    }
    ul = (NDIS_MAC_OPTION_COPY_LOOKAHEAD_DATA |
	  NDIS_MAC_OPTION_TRANSFERS_NOT_PEND |
	  NDIS_MAC_OPTION_NO_LOOPBACK);
    NdisMoveMemory(InformationBuffer, &ul, sizeof (ul));
    *BytesWritten = sizeof (ul);
    return NDIS_STATUS_SUCCESS;

  case OID_GEN_MEDIA_CONNECT_STATUS:
    if (InformationBufferLength < sizeof (ULONG)) {
      *BytesWritten = 0;
      *BytesNeeded = sizeof (ULONG);
      return NDIS_STATUS_INVALID_LENGTH;
    }
    if (adapter->is->link_state) {
      ul = NdisMediaStateConnected;
    }
    else {
      ul = NdisMediaStateDisconnected;
    }
    NdisMoveMemory(InformationBuffer, &ul, sizeof (ul));
    *BytesWritten = sizeof (ul);
    return NDIS_STATUS_SUCCESS;

  case OID_GEN_MAXIMUM_SEND_PACKETS:
    if (InformationBufferLength < sizeof (ULONG)) {
      *BytesWritten = 0;
      *BytesNeeded = sizeof (ULONG);
      return NDIS_STATUS_INVALID_LENGTH;
    }
    ul = NUM_TX / 2;
    NdisMoveMemory(InformationBuffer, &ul, sizeof (ul));
    *BytesWritten = sizeof (ul);
    return NDIS_STATUS_SUCCESS;

  case OID_GEN_XMIT_OK:
    if (InformationBufferLength < sizeof (ul)) {
      *BytesNeeded = sizeof (ul);
      *BytesWritten = 0;
      return NDIS_STATUS_INVALID_LENGTH;
    }
    ul = adapter->xmit_ok;
    NdisMoveMemory(InformationBuffer, &ul, sizeof (ul));
    *BytesNeeded = sizeof (ul);
    *BytesWritten = sizeof (ul);
    return NDIS_STATUS_SUCCESS;

  case OID_GEN_RCV_OK:
    if (InformationBufferLength < sizeof (ul)) {
      *BytesNeeded = sizeof (ul);
      *BytesWritten = 0;
      return NDIS_STATUS_INVALID_LENGTH;
    }
    ul = adapter->rcv_ok;
    NdisMoveMemory(InformationBuffer, &ul, sizeof (ul));
    *BytesNeeded = sizeof (ul);
    *BytesWritten = sizeof (ul);
    return NDIS_STATUS_SUCCESS;

  case OID_GEN_XMIT_ERROR:
    if (InformationBufferLength < sizeof (ULONG)) {
      *BytesWritten = 0;
      *BytesNeeded = sizeof (ULONG);
      return NDIS_STATUS_INVALID_LENGTH;
    }
    ul = adapter->xmit_error;
    NdisMoveMemory(InformationBuffer, &ul, sizeof (ul));
    *BytesWritten = sizeof (ul);
    return NDIS_STATUS_SUCCESS;

  case OID_GEN_RCV_ERROR:
    if (InformationBufferLength < sizeof (ULONG)) {
      *BytesWritten = 0;
      *BytesNeeded = sizeof (ULONG);
      return NDIS_STATUS_INVALID_LENGTH;
    }
    ul = adapter->rcv_error;
    NdisMoveMemory(InformationBuffer, &ul, sizeof (ul));
    *BytesWritten = sizeof (ul);
    return NDIS_STATUS_SUCCESS;

  case OID_GEN_RCV_NO_BUFFER:
    if (InformationBufferLength < sizeof (ULONG)) {
      *BytesWritten = 0;
      *BytesNeeded = sizeof (ULONG);
      return NDIS_STATUS_INVALID_LENGTH;
    }
    ul = adapter->rcv_no_buffer;
    NdisMoveMemory(InformationBuffer, &ul, sizeof (ul));
    *BytesWritten = sizeof (ul);
    return NDIS_STATUS_SUCCESS;

  case OID_802_3_PERMANENT_ADDRESS:
    if (InformationBufferLength < sizeof (macaddr)) {
      *BytesWritten = 0;
      *BytesNeeded = sizeof (macaddr);
      return NDIS_STATUS_INVALID_LENGTH;
    }
    NdisMoveMemory(InformationBuffer, macaddr, sizeof (macaddr));
    *BytesWritten = sizeof (macaddr);
    return NDIS_STATUS_SUCCESS;

  case OID_802_3_CURRENT_ADDRESS:
    if (InformationBufferLength < sizeof (macaddr)) {
      *BytesWritten = 0;
      *BytesNeeded = sizeof (macaddr);
      return NDIS_STATUS_INVALID_LENGTH;
    }
    NdisMoveMemory(InformationBuffer, macaddr, sizeof (macaddr));
    *BytesWritten = sizeof (macaddr);
    return NDIS_STATUS_SUCCESS;

  case OID_802_3_MULTICAST_LIST:
    return NDIS_STATUS_INVALID_OID;

  case OID_802_3_MAXIMUM_LIST_SIZE:
    if (InformationBufferLength < sizeof (ULONG)) {
      *BytesWritten = 0;
      *BytesNeeded = sizeof (ULONG);
      return NDIS_STATUS_INVALID_LENGTH;
    }
    ul = 32;
    NdisMoveMemory(InformationBuffer, &ul, sizeof (ul));
    *BytesWritten = sizeof (ul);
    return NDIS_STATUS_SUCCESS;

  case OID_802_3_MAC_OPTIONS:
    if (InformationBufferLength < sizeof (ULONG)) {
      *BytesWritten = 0;
      *BytesNeeded = sizeof (ULONG);
      return NDIS_STATUS_INVALID_LENGTH;
    }
    ul = 0;
    NdisMoveMemory(InformationBuffer, &ul, sizeof (ul));
    *BytesWritten = sizeof (ul);
    return NDIS_STATUS_SUCCESS;

  case OID_802_3_RCV_ERROR_ALIGNMENT:
    if (InformationBufferLength < sizeof (ULONG)) {
      *BytesWritten = 0;
      *BytesNeeded = sizeof (ULONG);
      return NDIS_STATUS_INVALID_LENGTH;
    }
    ul = adapter->rcv_error_alignment;
    NdisMoveMemory(InformationBuffer, &ul, sizeof (ul));
    *BytesWritten = sizeof (ul);
    return NDIS_STATUS_SUCCESS;

  case OID_802_3_XMIT_ONE_COLLISION:
    if (InformationBufferLength < sizeof (ULONG)) {
      *BytesWritten = 0;
      *BytesNeeded = sizeof (ULONG);
      return NDIS_STATUS_INVALID_LENGTH;
    }
    ul = adapter->xmit_one_collision;
    NdisMoveMemory(InformationBuffer, &ul, sizeof (ul));
    *BytesWritten = sizeof (ul);
    return NDIS_STATUS_SUCCESS;

  case OID_802_3_XMIT_MORE_COLLISIONS:
    if (InformationBufferLength < sizeof (ULONG)) {
      *BytesWritten = 0;
      *BytesNeeded = sizeof (ULONG);
      return NDIS_STATUS_INVALID_LENGTH;
    }
    ul = adapter->xmit_more_collisions;
    NdisMoveMemory(InformationBuffer, &ul, sizeof (ul));
    *BytesWritten = sizeof (ul);
    return NDIS_STATUS_SUCCESS;

  case OID_TCP_TASK_OFFLOAD:
    DbgPrint("Query task offload.\n");
    length = (sizeof (NDIS_TASK_OFFLOAD_HEADER) +
	      FIELD_OFFSET(NDIS_TASK_OFFLOAD, TaskBuffer) +
	      sizeof (NDIS_TASK_TCP_IP_CHECKSUM));
    if (length > InformationBufferLength) {
      *BytesWritten = 0;
      *BytesNeeded = length;
      return NDIS_STATUS_INVALID_LENGTH;
    }

    task_offload_header = (PNDIS_TASK_OFFLOAD_HEADER)InformationBuffer;
    if (task_offload_header->EncapsulationFormat.Encapsulation !=
	IEEE_802_3_Encapsulation) {
      DbgPrint("Encapsulation type not supported.\n");
      return NDIS_STATUS_NOT_SUPPORTED;
    }
    if (task_offload_header->Size != sizeof (NDIS_TASK_OFFLOAD_HEADER) ||
	task_offload_header->Version != NDIS_TASK_OFFLOAD_VERSION) {
      DbgPrint("Incorrect Size or Version.\n");
      return NDIS_STATUS_NOT_SUPPORTED;
    }

    task_offload_header->OffsetFirstTask = task_offload_header->Size;

    task_offload = (PNDIS_TASK_OFFLOAD)((char*)task_offload_header +
					task_offload_header->Size);
    task_offload->Size = g_task_offload.Size;
    task_offload->Version = g_task_offload.Version;
    task_offload->Task = g_task_offload.Task;
    task_offload->TaskBufferLength = g_task_offload.TaskBufferLength;
    task_offload->OffsetNextTask = 0;

    tcp_ip_checksum = (PNDIS_TASK_TCP_IP_CHECKSUM)task_offload->TaskBuffer;
    NdisMoveMemory(tcp_ip_checksum, &g_task_tcp_ip_checksum,
		   sizeof (g_task_tcp_ip_checksum));

    *BytesWritten = length;
    *BytesNeeded = 0;
    return NDIS_STATUS_SUCCESS;

  default:
    DbgPrint("unknown oid %x\n", (int)Oid);
    return NDIS_STATUS_NOT_SUPPORTED;
  }
}
/*}}}*/

/*{{{ set information */
NDIS_STATUS 
mx_set_information(IN NDIS_HANDLE MiniportAdapterContext,
		   IN NDIS_OID Oid,
		   IN PVOID InformationBuffer,
		   IN ULONG InformationBufferLength,
		   OUT PULONG BytesRead,
		   OUT PULONG BytesNeeded)
{
  mx_adapter_t *adapter = (mx_adapter_t*)MiniportAdapterContext;
  PNDIS_TASK_OFFLOAD_HEADER task_offload_header;
  PNDIS_TASK_OFFLOAD task_offload;
  PNDIS_TASK_TCP_IP_CHECKSUM task_tcp_ip_checksum;

  *BytesRead = 0;
  *BytesNeeded = 0;

  switch (Oid) {
    
  case OID_GEN_CURRENT_PACKET_FILTER:
#if 0
    if (InformationBufferLength < sizeof (ULONG)) {
      *BytesRead = 0;
      *BytesNeeded = sizeof (ULONG);
      return NDIS_STATUS_INVALID_LENGTH;
    }
    NdisMoveMemory(&ul, InformationBuffer, sizeof (ul));
    if (ul & ~MX_PACKET_FILTER) {
      return NDIS_STATUS_NOT_SUPPORTED;
    }
    adapter->filter = ul;
#endif
    return NDIS_STATUS_SUCCESS;

  case OID_GEN_CURRENT_LOOKAHEAD:
#if 0
    if (InformationBufferLength < sizeof (ULONG)) {
      *BytesRead = 0;
      *BytesNeeded = sizeof (ULONG);
      return NDIS_STATUS_INVALID_LENGTH;
    }
    NdisMoveMemory(&ul, InformationBuffer, sizeof (ul));
    if (ul > MX_LOOKAHEAD) {
      return NDIS_STATUS_NOT_SUPPORTED;
    }
    adapter->lookahead = ul;
#endif
    return NDIS_STATUS_SUCCESS;

  case OID_GEN_PROTOCOL_OPTIONS:
    /* TODO: ??? */
    return NDIS_STATUS_SUCCESS;

  case OID_802_3_MULTICAST_LIST:
    *BytesRead = 0;
    return NDIS_STATUS_MULTICAST_FULL;

  case OID_TCP_TASK_OFFLOAD:

    DbgPrint("Set task offload.\n");
    adapter->do_csum_offload = 0;
    adapter->do_xmit_tcp_csum = 0;
    adapter->do_xmit_udp_csum = 0;
    adapter->do_recv_tcp_csum = 0;
    adapter->do_recv_udp_csum = 0;

    if (InformationBufferLength < sizeof (NDIS_TASK_OFFLOAD_HEADER)) {
      return NDIS_STATUS_INVALID_LENGTH;
    }
    *BytesRead = sizeof (NDIS_TASK_OFFLOAD_HEADER);

    task_offload_header = (PNDIS_TASK_OFFLOAD_HEADER)InformationBuffer;
    if (task_offload_header->EncapsulationFormat.Encapsulation !=
	IEEE_802_3_Encapsulation) {
      task_offload_header->OffsetFirstTask = 0;
      return NDIS_STATUS_INVALID_DATA;
    }
    if (task_offload_header->OffsetFirstTask == 0) {
      DbgPrint("No offload task.\n");
      /* Still successful though. */
      return NDIS_STATUS_SUCCESS;
    }
    if (task_offload_header->OffsetFirstTask < task_offload_header->Size) {
      task_offload_header->OffsetFirstTask = 0;
      return NDIS_STATUS_FAILURE;
    }
    if (InformationBufferLength < (task_offload_header->OffsetFirstTask +
				   sizeof (NDIS_TASK_OFFLOAD))) {
      DbgPrint("Not enough room for one task.\n");
      return NDIS_STATUS_INVALID_LENGTH;
    }
    /* EUGENE: Cache EncapsulationFormat. */
    task_offload = (PNDIS_TASK_OFFLOAD)((char*)task_offload_header +
				   task_offload_header->OffsetFirstTask);
    *BytesRead += FIELD_OFFSET(NDIS_TASK_OFFLOAD, TaskBuffer);
    if (task_offload->Task != TcpIpChecksumNdisTask) {
      return NDIS_STATUS_NOT_SUPPORTED;
    }
    if (InformationBufferLength < (*BytesRead +
				   sizeof (NDIS_TASK_TCP_IP_CHECKSUM))) {
      *BytesNeeded = *BytesRead + sizeof (NDIS_TASK_TCP_IP_CHECKSUM);
      return NDIS_STATUS_INVALID_LENGTH;
    }
    if (g_task_offload.Task != task_offload->Task ||
	g_task_offload.Version != task_offload->Version) {
      DbgPrint("Wrong task or version.\n");
      return NDIS_STATUS_NOT_SUPPORTED;
    }
    task_tcp_ip_checksum = (PNDIS_TASK_TCP_IP_CHECKSUM)task_offload->TaskBuffer;
    /* V4Transmit */
    if (task_tcp_ip_checksum->V4Transmit.TcpChecksum) {
      DbgPrint("Requested V4Transmit.TcpChecksum.\n");
      if (!g_task_tcp_ip_checksum.V4Transmit.TcpChecksum) {
	DbgPrint("We don't support it.\n");
	return NDIS_STATUS_NOT_SUPPORTED;
      }
      adapter->do_csum_offload = 1;
      adapter->do_xmit_tcp_csum = 1;
    }
    if (task_tcp_ip_checksum->V4Transmit.UdpChecksum) {
      DbgPrint("Requested V4Transmit.UdpChecksum.\n");
      if (!g_task_tcp_ip_checksum.V4Transmit.UdpChecksum) {
	return NDIS_STATUS_NOT_SUPPORTED;
      }
      adapter->do_csum_offload = 1;
      adapter->do_xmit_udp_csum = 1;
    }
    if (task_tcp_ip_checksum->V4Transmit.IpChecksum) {
      DbgPrint("Requested V4Transmit.IpChecksum.\n");
      if (!g_task_tcp_ip_checksum.V4Transmit.IpChecksum) {
	DbgPrint("We don't support it.\n");
	return NDIS_STATUS_NOT_SUPPORTED;
      }
    }
    /* V4Receive */
    if (task_tcp_ip_checksum->V4Receive.TcpChecksum) {
      DbgPrint("Requested V4Receive.TcpChecksum.\n");
      if (!g_task_tcp_ip_checksum.V4Receive.TcpChecksum) {
	DbgPrint("We don't support it.\n");
	return NDIS_STATUS_NOT_SUPPORTED;
      }
      adapter->do_csum_offload = 1;
      adapter->do_recv_tcp_csum = 1;
    }
    if (task_tcp_ip_checksum->V4Receive.UdpChecksum) {
      DbgPrint("Requested V4Receive.UdpChecksum.\n");
      if (!g_task_tcp_ip_checksum.V4Receive.UdpChecksum) {
	DbgPrint("We don't support it.\n");
	return NDIS_STATUS_NOT_SUPPORTED;
      }
      adapter->do_csum_offload = 1;
      adapter->do_recv_udp_csum = 1;
    }
    if (task_tcp_ip_checksum->V4Receive.IpChecksum) {
      DbgPrint("Requested V4Receive.IpChecksum.\n");
      if (!g_task_tcp_ip_checksum.V4Receive.IpChecksum) {
	DbgPrint("We don't support it.\n");
	return NDIS_STATUS_NOT_SUPPORTED;
      }
    }
    /* V6 */
    if (task_tcp_ip_checksum->V6Transmit.TcpChecksum ||
	task_tcp_ip_checksum->V6Transmit.UdpChecksum ||
	task_tcp_ip_checksum->V6Receive.TcpChecksum ||
	task_tcp_ip_checksum->V6Receive.UdpChecksum) {
      DbgPrint("We don't support V6.\n");
      return NDIS_STATUS_NOT_SUPPORTED;
    }
    *BytesRead += sizeof (NDIS_TASK_TCP_IP_CHECKSUM);
    return NDIS_STATUS_SUCCESS;

  default:
#if 0
    MXE("unknown oid %x\n", (int)Oid);
#endif
    return NDIS_STATUS_INVALID_OID;
  }
}
/*}}}*/

#if MX_MUST_DO_4K_ALIGN
#define MX_RETURN_RECV_PACKET 1
#else
#define MX_RETURN_RECV_PACKET 0
#endif

VOID
mx_return_packet_helper(IN NDIS_HANDLE MiniportAdapterContext,
			IN PNDIS_PACKET Packet)
{
  mx_adapter_t *adapter = (mx_adapter_t*)MiniportAdapterContext;
  PVOID va;
  NDIS_PHYSICAL_ADDRESS pa;
  PNDIS_BUFFER buffer;
  int i;
  int idx;

  //DbgPrint("returning packet %p\n", Packet);
  /* TODO: Acquire recv lock? */
  //MXE("mx_return_packet\n");
  if (*MX_PACKET_IDX(Packet) == MX_SMALL_PACKET) {
    NdisQueryPacket(Packet, NULL, NULL, &buffer, NULL);
    va = (char*)NdisBufferVirtualAddress(buffer) - MX_MCP_ETHER_PAD;
    pa = *MX_PACKET_PA(Packet);
    i = adapter->is->ether->arch.next_small_to_fill;
#if 0
    DbgPrint("mx_return_packet next_small_to_fill = %d\n", i);
#endif
    idx = i & (NUM_RX - 1);
    adapter->is->ether->rx_small.info[idx].va = va;
    adapter->is->ether->rx_small.info[idx].pa = pa;
    adapter->is->ether->rx_small.info[idx].packet = Packet;
    adapter->is->ether->rx_small.info[idx].buffer = buffer;
    adapter->is->ether->rx_small.shadow[idx].addr_low = htonl(NdisGetPhysicalAddressLow(pa));
    adapter->is->ether->rx_small.shadow[idx].addr_high = htonl(NdisGetPhysicalAddressHigh(pa));
    adapter->is->ether->arch.next_small_to_fill++;

#if MX_RETURN_RECV_PACKET
    mx_pio_memcpy(&adapter->is->ether->rx_small.ring[idx],
		  &adapter->is->ether->rx_small.shadow[idx],
		  sizeof (*adapter->is->ether->rx_small.ring), 0);
    MX_STBAR();
    MX_PIO_WRITE(adapter->is->ether->rx_small.lanai_cnt,
		 htonl(adapter->is->ether->arch.next_small_to_fill));
#else
    /* copy 8 descriptors to the mcp at a time */
    if ((idx & 3) == 3) {
      /* FIXME: avoid 64 bytes write-combining */
      mx_pio_memcpy(&adapter->is->ether->rx_small.ring[idx - 3],
		  &adapter->is->ether->rx_small.shadow[idx - 3],
		  4 * sizeof (*adapter->is->ether->rx_small.ring), 0);
      MX_STBAR();
      MX_PIO_WRITE(adapter->is->ether->rx_small.lanai_cnt,
		   htonl(adapter->is->ether->arch.next_small_to_fill));
    }
#endif
  }
  else if (*MX_PACKET_IDX(Packet) == MX_BIG_PACKET) {
    NdisQueryPacket(Packet, NULL, NULL, &buffer, NULL);
    va = (char*)NdisBufferVirtualAddress(buffer) - MX_MCP_ETHER_PAD;
    pa = *MX_PACKET_PA(Packet);
    i = adapter->is->ether->arch.next_big_to_fill;
#if 0
    DbgPrint("mx_return_packet next_big_to_fill = %d\n", i);
#endif
    idx = i & (NUM_RX - 1);
    adapter->is->ether->rx_big.info[idx].va = va;
    adapter->is->ether->rx_big.info[idx].pa = pa;
    adapter->is->ether->rx_big.info[idx].packet = Packet;
    adapter->is->ether->rx_big.info[idx].buffer = buffer;
    adapter->is->ether->rx_big.shadow[idx].addr_low = htonl(NdisGetPhysicalAddressLow(pa));
    adapter->is->ether->rx_big.shadow[idx].addr_high = htonl(NdisGetPhysicalAddressHigh(pa));
    adapter->is->ether->arch.next_big_to_fill++;

#if MX_RETURN_RECV_PACKET
    mx_pio_memcpy(&adapter->is->ether->rx_big.ring[idx],
		  &adapter->is->ether->rx_big.shadow[idx],
		  sizeof (*adapter->is->ether->rx_big.ring), 0);
    MX_STBAR();
    MX_PIO_WRITE(adapter->is->ether->rx_big.lanai_cnt,
		 htonl(adapter->is->ether->arch.next_big_to_fill));
#else
    /* copy 8 descriptors to the mcp at a time */
    if ((idx & 3) == 3) {
      /* FIXME: avoid 64 bytes write-combining */
      mx_pio_memcpy(&adapter->is->ether->rx_big.ring[idx - 3],
		  &adapter->is->ether->rx_big.shadow[idx - 3],
		  4 * sizeof (*adapter->is->ether->rx_big.ring), 0);
      MX_STBAR();
      MX_PIO_WRITE(adapter->is->ether->rx_big.lanai_cnt,
		   htonl(adapter->is->ether->arch.next_big_to_fill));
    }
#endif
  }
  else {
    DbgPrint("unknown packet!!!\n");
  }
}

VOID
mx_return_packet(IN NDIS_HANDLE MiniportAdapterContext,
		 IN PNDIS_PACKET Packet)
{
  mx_adapter_t *adapter = (mx_adapter_t*)MiniportAdapterContext;

  NdisAcquireSpinLock(&adapter->recv_lock);
  mx_return_packet_helper(MiniportAdapterContext, Packet);
  NdisReleaseSpinLock(&adapter->recv_lock);
}

typedef struct tcphdr_t
{
  uint16_t source_port;
  uint16_t dest_port;
  uint32_t seq_num;
  uint32_t ack_num;
  uint16_t flags;
  uint16_t window;
  uint16_t csum;
  uint16_t urgent;
} tcphdr_t;

typedef struct udphdr_t
{
  uint16_t source_port;
  uint16_t dest_port;
  uint16_t length;
  uint16_t csum;
} udphdr_t;

void
dump_packet(PNDIS_PACKET packet, int is_recv)
{
#if 0
  PNDIS_BUFFER buffer;
  UINT buffer_count;
  UINT total_packet_length;
  UINT dump_length;
  UINT count;
  uint8_t *va;
  UINT buffer_length;

  if (is_recv) {
    DbgPrint("receive packed\n");
  }
  else {
    DbgPrint("sending packet\n");
  }

  NdisQueryPacket(packet, NULL, &buffer_count, &buffer,
		  &total_packet_length);
  DbgPrint("length %u\n", total_packet_length);
  if (total_packet_length > 64) {
    dump_length = 64;
  }
  else {
    dump_length = total_packet_length;
  }

  count = 0;
  NdisQueryBuffer(buffer, &va, &buffer_length);
  while (count < dump_length) {
    if (buffer_length == 0) {
      NdisGetNextBuffer(buffer, &buffer);
      if (buffer != NULL) {
	NdisQueryBuffer(buffer, &va, &buffer_length);
      }
      else {
	break;
      }
    }

    DbgPrint("%02x ", (unsigned int)*va);
    va++;
    count++;
    if ((count % 16) == 0) {
      DbgPrint("\n");
    }
    buffer_length--;
  }
  DbgPrint("\n");
#endif
}


NDIS_STATUS
mx_send_packet(mx_adapter_t *adapter, PNDIS_PACKET packet, int fromq)
{
  UINT phys_buffer_count;
  UINT buffer_count;
  PNDIS_BUFFER buffer;
  PNDIS_BUFFER buffer2; /* just to get ip header length */
  int byte_count;
  int ip_hlv;
  UINT packet_length;
  PSCATTER_GATHER_LIST sg_list;
  UINT loop;
  struct mx_ether *eth;
  mcp_kreq_ether_send_t req_list[MX_MCP_ETHER_MAX_SEND_FRAG+1];
  mcp_kreq_ether_send_t *req;
  char *header;
  UINT len;
  char *p;
  PVOID buf_va;
  UINT buf_len;
  int i, idx, avail, frag_cnt;
  NDIS_TCP_IP_CHECKSUM_PACKET_INFO checksum;
  uint16_t cksum_offset;
  uint16_t pseudo_hdr_offset;

  //DbgPrint("sending packet %p\n", packet);
  //dump_packet(packet, 0);

  eth = adapter->is->ether;
  avail = NUM_TX - 2 - (eth->tx.req - eth->tx.done);
  NdisQueryPacket(packet, &phys_buffer_count, &buffer_count, &buffer,
		  &packet_length);

  if (adapter->do_csum_offload) {
    buffer2 = buffer;
    byte_count = 14;
    NdisQueryBuffer(buffer2, &buf_va, &buf_len);
    while (buf_len <= byte_count) {
      byte_count -= buf_len;
      NdisGetNextBuffer(buffer2, &buffer2);
      NdisQueryBuffer(buffer2, &buf_va, &buf_len);
    }
    ip_hlv = ((unsigned char*)buf_va)[byte_count];
    ip_hlv = (ip_hlv & 0x0F) << 2;

    checksum.Value = PtrToUlong(NDIS_PER_PACKET_INFO_FROM_PACKET(packet, TcpIpChecksumPacketInfo));
  }

  sg_list = NDIS_PER_PACKET_INFO_FROM_PACKET(packet,
					     ScatterGatherListPacketInfo);
  frag_cnt = sg_list->NumberOfElements-1;
  if (avail < 4) {
    if (fromq) {
      /* Put it back at the front. */
      STAILQ_INSERT_HEAD(&adapter->sendq, MX_PACKET_Q(packet), entries);
      
    }
    else {
      STAILQ_INSERT_TAIL(&adapter->sendq, MX_PACKET_Q(packet), entries);
    }
    return NDIS_STATUS_PENDING;
  }

  i = eth->tx.req;
  idx = i & (NUM_TX - 1);
  eth->tx.info[idx].packet = NULL; /* header doesn't count */
  req = req_list;
  /* next one is the whole packet since we are coalescing right now */
  i++;
  idx = i & (NUM_TX - 1);
  eth->tx.info[idx].packet = packet;

  if (MX_MUST_DO_4K_ALIGN ||
      avail < (1 + 1 + frag_cnt) ||
      (1 + frag_cnt) > MX_MCP_ETHER_MAX_SEND_FRAG ||
      packet_length < 60) {
    p = eth->tx.info[idx].va;
    while (buffer != NULL) {
      NdisQueryBuffer(buffer, &buf_va, &buf_len);
      NdisMoveMemory(p, buf_va, buf_len);
      p += buf_len;
      NdisGetNextBuffer(buffer, &buffer);
    }
    if (packet_length < 60) {
      NdisZeroMemory(p, 60 - packet_length);
      packet_length = 60;
    }

    req->head.pseudo_hdr_offset = 0;
    req->head.cksum_offset = 0;
    NdisMoveMemory(&req->head.dest_high16, &((char*)eth->tx.info[idx].va)[0],
		   sizeof (req->head.dest_high16));
    NdisMoveMemory(&req->head.dest_low32, &((char*)eth->tx.info[idx].va)[2],
		   sizeof (req->head.dest_low32));
    req->head.flags = MX_MCP_ETHER_FLAGS_VALID | MX_MCP_ETHER_FLAGS_HEAD;

    if (adapter->do_csum_offload) {
      if (checksum.Transmit.NdisPacketChecksumV4) {
	cksum_offset = 14 + ip_hlv;
	if (checksum.Transmit.NdisPacketTcpChecksum &&
	    adapter->do_xmit_tcp_csum ) {
	  pseudo_hdr_offset = cksum_offset + offsetof(struct tcphdr_t, csum);
	  req->head.pseudo_hdr_offset = htons(pseudo_hdr_offset);
	  req->head.cksum_offset = htons(cksum_offset);
	  req->head.flags |= MX_MCP_ETHER_FLAGS_CKSUM;
	}
	else if (checksum.Transmit.NdisPacketUdpChecksum &&
		 adapter->do_xmit_udp_csum) {
	  pseudo_hdr_offset = cksum_offset + offsetof(struct udphdr_t, csum);
	  req->head.pseudo_hdr_offset = htons(pseudo_hdr_offset);
	  req->head.cksum_offset = htons(cksum_offset);
	  req->head.flags |= MX_MCP_ETHER_FLAGS_CKSUM;
	}
      }
    }

    req++;
    req->frag.addr_low = htonl(NdisGetPhysicalAddressLow(eth->tx.info[idx].pa));
    req->frag.addr_high = htonl(NdisGetPhysicalAddressHigh(eth->tx.info[idx].pa));
    req->frag.length = htons(packet_length);
    req->frag.flags = MX_MCP_ETHER_FLAGS_VALID | MX_MCP_ETHER_FLAGS_LAST;

    frag_cnt = 0;
  }
  else {  
    for (loop = 0; loop < sg_list->NumberOfElements; ++loop) {
      if (loop == 0) {
	NdisQueryBufferSafe(buffer, &header, &len, LowPagePriority);
	if (header == NULL) {
	  DbgPrint("query buffer failed\n!!!\n");
	  DbgBreakPoint();
	}
	NdisMoveMemory(&req->head.dest_high16, &header[0],
		       sizeof (req->head.dest_high16));
	NdisMoveMemory(&req->head.dest_low32, &header[2],
		       sizeof (req->head.dest_low32));
	req->head.flags = MX_MCP_ETHER_FLAGS_VALID | MX_MCP_ETHER_FLAGS_HEAD;
	req->head.pseudo_hdr_offset = 0;
	req->head.cksum_offset = 0;

	if (adapter->do_csum_offload) {
	  if (checksum.Transmit.NdisPacketChecksumV4) {
	    cksum_offset = 14 + ip_hlv;
	    if (checksum.Transmit.NdisPacketTcpChecksum &&
		adapter->do_xmit_tcp_csum) {
	      pseudo_hdr_offset = cksum_offset + offsetof(struct tcphdr_t, csum);
	      req->head.pseudo_hdr_offset = htons(pseudo_hdr_offset);
	      req->head.cksum_offset = htons(cksum_offset);
	      req->head.flags |= MX_MCP_ETHER_FLAGS_CKSUM;
	    }
	    else if (checksum.Transmit.NdisPacketUdpChecksum &&
		     adapter->do_xmit_udp_csum) {
	      pseudo_hdr_offset = cksum_offset + offsetof(struct udphdr_t, csum);
	      req->head.pseudo_hdr_offset = htons(pseudo_hdr_offset);
	      req->head.cksum_offset = htons(cksum_offset);
	      req->head.flags |= MX_MCP_ETHER_FLAGS_CKSUM;
	    }
	  }
	}

	req++;
      }
      req->frag.addr_low = htonl(NdisGetPhysicalAddressLow(sg_list->Elements[loop].Address));
      req->frag.addr_high = htonl(NdisGetPhysicalAddressHigh(sg_list->Elements[loop].Address));
      req->frag.length = htons((USHORT)sg_list->Elements[loop].Length);
      req->frag.flags = MX_MCP_ETHER_FLAGS_VALID;
      req++;
    }
    req--;
    req->frag.flags |= MX_MCP_ETHER_FLAGS_LAST;
  }

  //DbgPrint("frag_cnt = %d\n", (int)frag_cnt);
  mx_ether_submit_tx_req(eth, req_list, 1 + 1 + frag_cnt);

  return NDIS_STATUS_SUCCESS;
}

VOID
mx_send_packets(IN NDIS_HANDLE MiniportAdapterContext,
		IN PPNDIS_PACKET PacketArray,
		IN UINT NumberOfPackets)
{
#if MX_ENABLE_ETHERNET
  mx_adapter_t *adapter = (mx_adapter_t*)MiniportAdapterContext;
  UINT i;

  //MXE("mx_send_packets\n");
  NdisAcquireSpinLock(&adapter->send_lock);
  for (i = 0; i < NumberOfPackets; ++i) {
    if (!STAILQ_EMPTY(&adapter->sendq)) {
      STAILQ_INSERT_TAIL(&adapter->sendq, MX_PACKET_Q(PacketArray[i]),
			 entries);
    }
    else {
      mx_send_packet(adapter, PacketArray[i], FALSE);
    }
  }
  NdisReleaseSpinLock(&adapter->send_lock);
#else
  mx_adapter_t *adapter = (mx_adapter_t*)MiniportAdapterContext;
  UINT i;

  MXE("mx_send_packets\n");

  for (i = 0; i < NumberOfPackets; ++i) {
    NDIS_SET_PACKET_STATUS(PacketArray[i], NDIS_STATUS_SUCCESS);
    NdisMSendComplete(adapter->miniport_adapter_handle, PacketArray[i],
		      NDIS_STATUS_SUCCESS);
  }
#endif
}

NDIS_STATUS
mx_reset(OUT PBOOLEAN AddressingReset,
	 IN NDIS_HANDLE MiniportAdapterContext)
{
  AddressingReset = FALSE;
  return NDIS_STATUS_SUCCESS;
}

BOOLEAN
mx_check_for_hang(IN NDIS_HANDLE MiniportAdapterContext)
{
  return FALSE;
}

VOID
mx_unload(IN PDRIVER_OBJECT DriverObject)
{
  MXE("---> mx_unload\n");
  /* TODO: Only call if init succeeded? */
  mx_unit_free();
  mx_spin_lock_destroy(&g_dev_lock);
  mx_finalize_driver();
  MXE("<--- mx_unload\n");
}

static PDRIVER_DISPATCH ndis_pnp = 0;
static int g_mx_usage_count = 0;

NTSTATUS
mx_mj_create(IN PDEVICE_OBJECT DeviceObject, IN PIRP Irp)
{
  PIO_STACK_LOCATION irp_stack;
  NTSTATUS status = STATUS_SUCCESS;
  UNICODE_STRING ustr;
  ULONG ul;
  int idx;
  mx_endpt_state_t *endpt;
  int can_be_privileged;

  DbgPrint("mx_mj_create\n");

  can_be_privileged = mx_can_be_privileged(Irp);
  DbgPrint("can_be_privileged = %d\n", can_be_privileged);

  irp_stack = IoGetCurrentIrpStackLocation(Irp);
  idx = mx_unit_lookup(DeviceObject);
  if (idx == -1) { /* mxctl */
    if (irp_stack->FileObject->FileName.Length == 0) {
      DbgPrint("mxctl\n");
      irp_stack->FileObject->FsContext = NULL;
      irp_stack->FileObject->FsContext2 = NULL;
    }
    else if (wcscmp(irp_stack->FileObject->FileName.Buffer, L"\\p") == 0) {
      DbgPrint("mxctlp\n");
      if (can_be_privileged) {
	irp_stack->FileObject->FsContext = NULL;
	irp_stack->FileObject->FsContext2 = NULL;
      }
      else {
	status = STATUS_UNSUCCESSFUL;
      }
    }
    else {
      status = STATUS_UNSUCCESSFUL;
    }
  }
  else if (irp_stack->FileObject->FileName.Length == 0) {
    DbgPrint("mx%d\n", idx);
    irp_stack->FileObject->FsContext = g_instances[idx];
    irp_stack->FileObject->FsContext2 = NULL;
  }
  else if (wcscmp(irp_stack->FileObject->FileName.Buffer, L"\\p") == 0) {
    /* TODO: Make sure you have permission. */
    DbgPrint("mxp%d\n", idx);
    if (can_be_privileged) {
      irp_stack->FileObject->FsContext = g_instances[idx];
      irp_stack->FileObject->FsContext2 = NULL;
    }
    else {
      status = STATUS_UNSUCCESSFUL;
    }
  }
  else if (wcscmp(irp_stack->FileObject->FileName.Buffer, L"\\r") == 0) {
    /* TODO: Make sure you have permission. */
    DbgPrint("mxr%d\n", idx);
    if (can_be_privileged) {
      endpt = mx_kmalloc(sizeof (*endpt), MX_MZERO);
      if (endpt != NULL) {
	irp_stack->FileObject->FsContext = g_instances[idx];
	irp_stack->FileObject->FsContext2 = endpt;
	endpt->flags = MX_ES_RAW;
	endpt->endpt = -1;
	/* TODO: See if is->raw.use_count != 0? */
      }
      else {
	status = STATUS_UNSUCCESSFUL;
      }
    }
    else {
      status = STATUS_UNSUCCESSFUL;
    }
  }
  else if (wcsncmp(irp_stack->FileObject->FileName.Buffer, L"\\e",
		   wcslen(L"\\e")) == 0) {
    ustr.Length = (irp_stack->FileObject->FileName.Length -
		   (2 * sizeof (wchar_t)));
    ustr.MaximumLength = (irp_stack->FileObject->FileName.MaximumLength -
			  (2 * sizeof (wchar_t)));
    ustr.Buffer = &irp_stack->FileObject->FileName.Buffer[2];
    status = RtlUnicodeStringToInteger(&ustr, 10, &ul);
    if (status == STATUS_SUCCESS) {
      DbgPrint("mx%de%d\n", idx, (int)ul);
      endpt = mx_kmalloc(sizeof (*endpt), MX_MZERO);
      if (endpt != NULL) {
	/* TODO: Wrap up setting of endpoint in function? */
        mx_spin_lock(&g_dev_lock);
	/* TODO: See if is->es[ul] is occupied? */
	if (g_endpts[idx * mx_max_endpoints + ul] == NULL) {
	  g_endpts[idx * mx_max_endpoints + ul] = endpt;
	  endpt->endpt = ul;
	  endpt->arch.endpoint = ul;
	  irp_stack->FileObject->FsContext = g_instances[idx];
	  irp_stack->FileObject->FsContext2 = endpt;
	}
	else {
	  status = STATUS_UNSUCCESSFUL;
	  mx_kfree(endpt);
	}
	mx_spin_unlock(&g_dev_lock);
      }
      else {
	status = STATUS_UNSUCCESSFUL;
      }
    }
  }
  else {
    status = STATUS_UNSUCCESSFUL;
  }
  if (status == STATUS_SUCCESS) {
    ++g_mx_usage_count;
  }
  Irp->IoStatus.Information = 0;
  Irp->IoStatus.Status = status;
  IoCompleteRequest(Irp, IO_NO_INCREMENT);

  return status;
} 

NTSTATUS
mx_mj_cleanup(IN PDEVICE_OBJECT DeviceObject, IN PIRP Irp)
{
  PIO_STACK_LOCATION irp_stack;
  NTSTATUS status = STATUS_SUCCESS;
  
  DbgPrint("mx_mj_cleanup\n");

  irp_stack = IoGetCurrentIrpStackLocation(Irp);
  
  Irp->IoStatus.Information = 0;
  Irp->IoStatus.Status = status;
  IoCompleteRequest(Irp, IO_NO_INCREMENT);

  return status;
} 

NTSTATUS
mx_mj_close(IN PDEVICE_OBJECT DeviceObject, IN PIRP Irp)
{
  PIO_STACK_LOCATION irp_stack;
  NTSTATUS status = STATUS_SUCCESS;
  mx_instance_state_t *is;
  mx_endpt_state_t *es;
  int idx;
  int privileged;
  int endpt;

  DbgPrint("mx_mj_close\n");
  
  irp_stack = IoGetCurrentIrpStackLocation(Irp);
  mx_decode_context(irp_stack->FileObject, &is, &es, &privileged);
  if (is == NULL) { /* mxctl */
    /* Does anything need to be done? */
  } else {
    idx = is->id;
    if (es != NULL) {
      endpt = es->endpt;
    }
    else {
      endpt = -1;
    }
    DbgPrint("unit %d, privileged %d, endpoint %d\n", idx, privileged, endpt);
    if (endpt != -1) {
      mx_spin_lock(&g_dev_lock);
      mx_assert(g_endpts[idx * mx_max_endpoints + endpt] == es);
      g_endpts[idx * mx_max_endpoints + endpt] = NULL;
      mx_spin_unlock(&g_dev_lock);
    }
    if (es != NULL) {
      if (es->is != NULL) {
	mx_munmapall(es);
	mx_common_close(es);
      }
      mx_kfree(es);
    }
  }

  --g_mx_usage_count;
  Irp->IoStatus.Information = 0;
  Irp->IoStatus.Status = status;
  IoCompleteRequest(Irp, IO_NO_INCREMENT);

  return status;
} 

NTSTATUS
mx_mj_device_control(IN PDEVICE_OBJECT DeviceObject, IN PIRP Irp)
{
  PIO_STACK_LOCATION irp_stack;
  NTSTATUS status = STATUS_SUCCESS;
  void *in, *out;
  ULONG in_len, out_len;
  char buff[80];
  ULONG loop;
  int privileged;
  uint32_t cmd;
  int retval;
  mx_instance_state_t *is;
  mx_endpt_state_t *es;
  
  irp_stack = IoGetCurrentIrpStackLocation(Irp);
  in = irp_stack->Parameters.DeviceIoControl.Type3InputBuffer;
  out = Irp->UserBuffer;
  in_len = irp_stack->Parameters.DeviceIoControl.InputBufferLength;
  out_len = irp_stack->Parameters.DeviceIoControl.OutputBufferLength;
  cmd = irp_stack->Parameters.DeviceIoControl.IoControlCode;
  mx_decode_context(irp_stack->FileObject, &is, &es, &privileged);

  switch (irp_stack->Parameters.DeviceIoControl.IoControlCode) {
  case MX_IOCTL_FOO:
    DbgPrint("foo\n");
    // only called from user space ?
    mx_arch_copyin((mx_uaddr_t)in, buff, in_len);
    for (loop = 0; loop < in_len; ++loop) {
      buff[loop]++;
    }
    mx_arch_copyout(buff, (mx_uaddr_t)out, in_len);
    break;
  case MX_IOCTL_BAR:
    DbgPrint("bar\n");
    /*
    mx_common_ioctl(es,
		    irp_stack->Parameters.DeviceIoControl.IoControlCode,
		    in, in_len, out, out_len, 0);
    mx_endptless_ioctl(irp_stack->Parameters.DeviceIoControl.IoControlCode,
		       in, privileged, in_len, out, out_len, 0, 0);
    */
    break;
  case MX_IOCTL_HANG1: mx_hang_test(1); break;
  case MX_IOCTL_HANG2: mx_hang_test(2); break;
  case MX_IOCTL_HANG3: mx_hang_test(3); break;
  case MX_IOCTL_HANG4: mx_hang_test(4); break;
  case MX_IOCTL_HANG5: mx_hang_test(5); break;
  case MX_IOCTL_HANG6: mx_hang_test(6); break;
  case MX_IOCTL_HANG7: mx_hang_test(7); break;
  case MX_IOCTL_HANG8: mx_hang_test(8); break;
  default:
    if (es == NULL) {
      retval = mx_endptless_ioctl(cmd, (mx_uaddr_t)in, privileged, 0);
    }
    else {
      if (cmd == MX_SET_ENDPOINT) {
	retval = mx_set_endpoint(es, is->id, privileged, es->endpt,
				 (mx_uaddr_t)in, 0);
      }
      else if (cmd == MX_SET_RAW) {
	retval = mx_set_endpoint(es, is->id, privileged, es->endpt,
				 (mx_uaddr_t)in, 1);
      }
      else {
	mx_mutex_enter(&es->sync);
	es->ref_count++;
	mx_mutex_exit(&es->sync);
	if (cmd == MX_MMAP) {
	  retval = mx_mmap(es, (mx_uaddr_t)in);
	}
	else {
	  retval = mx_common_ioctl(es, cmd, (mx_uaddr_t)in);
	  if (retval == ENOTTY) {
	    retval = mx_endptless_ioctl(cmd, (mx_uaddr_t)in, privileged, 0);
	  }
	}
	mx_mutex_enter(&es->sync);
	es->ref_count--;
	mx_mutex_exit(&es->sync);
      }
    }
    if (retval != 0) {
      /* TODO: Magic number. */
      /* 29 - customer bit */
      /* 30 - 31 - severity (success, information, warning, error) */
      status = retval | (1 << 29) | (1 << 30) | (1 << 31);
    }
  }
      
  Irp->IoStatus.Information = 0;
  Irp->IoStatus.Status = status;
  IoCompleteRequest(Irp, IO_NO_INCREMENT);

  return status;
} 

NTSTATUS
mx_mj_pnp(IN PDEVICE_OBJECT DeviceObject, IN PIRP Irp)
{
  NTSTATUS status;
  PIO_STACK_LOCATION irp_stack;

  irp_stack = IoGetCurrentIrpStackLocation(Irp);
  if (irp_stack->MinorFunction == IRP_MN_QUERY_REMOVE_DEVICE &&
      g_mx_usage_count > 0) {
    status = STATUS_UNSUCCESSFUL;
    Irp->IoStatus.Status = status;
    Irp->IoStatus.Information = 0;
    IoCompleteRequest(Irp, IO_NO_INCREMENT);
  }
  else {
    status = ndis_pnp(DeviceObject, Irp);
  }
  return status;
}

VOID
mx_isr(OUT PBOOLEAN InterruptRecognized,
       OUT PBOOLEAN QueueMiniportHandleInterrupt,
       IN NDIS_HANDLE MiniportAdapterContext)
{
  mx_adapter_t *adapter = (mx_adapter_t*)MiniportAdapterContext;
  int type;
#if MX_DRIVER_API_MAGIC < 0x500
  int intrq;
#endif

#if MX_DRIVER_API_MAGIC >= 0x500
  if (adapter->is->intr.ring_mask == 0) {
#else
  if (adapter->is->intr.maxslots == 0) {
#endif
    *InterruptRecognized = FALSE;
    *QueueMiniportHandleInterrupt = FALSE;
    return;
  }

#if MX_DRIVER_API_MAGIC >= 0x500
  type = mx_intr_slot(adapter->is, adapter->is->intr.slot)->type;
#else
  intrq = adapter->is->intr.intrq;
  type = adapter->is->intr.q[intrq][adapter->is->intr.slot].type;
#endif
  if (type != 0) {
    if (!adapter->is->arch.intr_pending) {
      adapter->is->arch.intr_pending = 1;
      if (adapter->is->board_ops.disable_interrupt != NULL) {
	adapter->is->board_ops.disable_interrupt(adapter->is);
      }
      *InterruptRecognized = TRUE;
      *QueueMiniportHandleInterrupt = TRUE;
    }
    else {
      //DbgPrint("already queued\n");
      *InterruptRecognized = TRUE;
      *QueueMiniportHandleInterrupt = FALSE;
    }
  } else {
    *InterruptRecognized = FALSE;
    *QueueMiniportHandleInterrupt = FALSE;
  }
}

BOOLEAN
mx_intr_sync_func(IN PVOID context)
{
  mx_adapter_t *adapter = (mx_adapter_t*)context;

  if (adapter->is->arch.intr_pending == 0) {
    DbgPrint("intr_pending == 0?\n");
  }
  adapter->is->arch.intr_pending = 0;
  if (adapter->is->board_ops.enable_interrupt != NULL) {
    adapter->is->board_ops.enable_interrupt(adapter->is);
  }
  return 0;
}

VOID
mx_handle_interrupt(IN NDIS_HANDLE MiniportAdapterContext)
{
  mx_adapter_t *adapter = (mx_adapter_t*)MiniportAdapterContext;
  static int in_handle_interrupt = 0;

  mx_spin_lock(&adapter->is->arch.intr_lock);
  in_handle_interrupt++;
  if (in_handle_interrupt > 1) {
    DbgPrint("in_handle_interrupt = %d?\n", in_handle_interrupt);
  }
  mx_common_interrupt(adapter->is);
  in_handle_interrupt--;
  mx_spin_unlock(&adapter->is->arch.intr_lock);
  NdisMSynchronizeWithInterrupt(&adapter->is->arch.miniport_interrupt,
				mx_intr_sync_func, adapter);
#if 0
  if (adapter->is->board_ops.enable_interrupt != NULL) {
    adapter->is->board_ops.enable_interrupt(adapter->is);
  }
#endif
}

void
mx_read_module_param(HANDLE h, PCWSTR src_str, uint32_t *val)
{
  ULONG scratch[8];
  KEY_VALUE_PARTIAL_INFORMATION *pinfo;
  UNICODE_STRING str;
  NTSTATUS ntstatus;
  ULONG out_len;
  uint32_t tmp;

  RtlInitUnicodeString(&str, src_str);
  pinfo = (KEY_VALUE_PARTIAL_INFORMATION*)scratch;

  ntstatus = ZwQueryValueKey(h, &str, KeyValuePartialInformation, pinfo,
			     sizeof (scratch), &out_len);
  if (ntstatus == STATUS_SUCCESS &&
      pinfo->Type == REG_DWORD &&
      out_len <= sizeof (scratch)) {
    RtlCopyMemory(&tmp, &pinfo->Data, sizeof (tmp));
    *val = tmp;
  }
}

void
mx_read_module_params(PUNICODE_STRING reg_path)
{
  HANDLE h;
  OBJECT_ATTRIBUTES oa;
  NTSTATUS ntstatus;

  InitializeObjectAttributes(&oa, reg_path, OBJ_CASE_INSENSITIVE,
			     NULL, NULL);
  ntstatus = ZwOpenKey(&h, KEY_READ, &oa);
  if (NT_ERROR(ntstatus)) {
    goto abort_with_nothing;
  }

  DbgPrint("mx_debug_mask = %x\n", mx_debug_mask);
  DbgPrint("mx_max_instance = %u\n", mx_max_instance);
  DbgPrint("mx_max_nodes = %u\n", mx_max_nodes);
  DbgPrint("mx_max_endpoints = %u\n", mx_max_endpoints);

  mx_read_module_param(h, L"mx_debug_mask", &mx_debug_mask);
  mx_read_module_param(h, L"mx_max_instance", &mx_max_instance);
  mx_read_module_param(h, L"mx_max_nodes", &mx_max_nodes);
  mx_read_module_param(h, L"mx_max_endpoints", &mx_max_endpoints);

  DbgPrint("mx_debug_mask = %x\n", mx_debug_mask);
  DbgPrint("mx_max_instance = %u\n", mx_max_instance);
  DbgPrint("mx_max_nodes = %u\n", mx_max_nodes);
  DbgPrint("mx_max_endpoints = %u\n", mx_max_endpoints);

  ZwClose(h);

 abort_with_nothing:
  return;
}

NDIS_STATUS
DriverEntry(IN PDRIVER_OBJECT DriverObject,
	    IN PUNICODE_STRING RegistryPath)
{
  NDIS_STATUS status;
  NDIS_MINIPORT_CHARACTERISTICS mini_char;
  NDIS_STRING dev_name;
  NDIS_STRING sym_name;
  PDRIVER_DISPATCH mj_func[IRP_MJ_MAXIMUM_FUNCTION+1];

  DbgPrint("hello world\n");
  MXE("---> DriverEntry\n");
  //DbgBreakPoint();

  mx_read_module_params(RegistryPath);

  NdisMInitializeWrapper(&g_ndis_wrapper, DriverObject, RegistryPath,
			 NULL);
  NdisZeroMemory(&mini_char, sizeof (mini_char));
  mini_char.MajorNdisVersion = 5;
  mini_char.MinorNdisVersion = 0;
  mini_char.HaltHandler = mx_halt;
  mini_char.InitializeHandler = mx_initialize;
  mini_char.QueryInformationHandler = mx_query_information;
  mini_char.SetInformationHandler = mx_set_information;
  mini_char.ReturnPacketHandler = mx_return_packet;
  mini_char.SendPacketsHandler = mx_send_packets;
  mini_char.ResetHandler = mx_reset;
  mini_char.CheckForHangHandler = mx_check_for_hang;
  mini_char.HandleInterruptHandler = mx_handle_interrupt;
  mini_char.ISRHandler = mx_isr;

  status = NdisMRegisterMiniport(g_ndis_wrapper, &mini_char,
				 sizeof (mini_char));
  if (status != NDIS_STATUS_SUCCESS) {
    goto abort_with_wrapper;
  }
  NdisMRegisterUnloadHandler(g_ndis_wrapper, mx_unload);

  if (mx_init_driver() != 0) {
    status = NDIS_STATUS_FAILURE;
    goto abort_with_wrapper;
  }

  mx_spin_lock_init(&g_dev_lock, NULL, 0, NULL);
  if (mx_unit_alloc() != 0) {
    status = NDIS_STATUS_FAILURE;
    goto abort_with_dev_lock;
  }

  NdisInitUnicodeString(&dev_name, L"\\Device\\mxctl");
  NdisInitUnicodeString(&sym_name, L"\\DosDevices\\mxctl");
  NdisZeroMemory(mj_func, (IRP_MJ_MAXIMUM_FUNCTION+1) * sizeof (PDRIVER_DISPATCH));
  mj_func[IRP_MJ_CREATE] = mx_mj_create;
  mj_func[IRP_MJ_CLEANUP] = mx_mj_cleanup;
  mj_func[IRP_MJ_CLOSE] = mx_mj_close;
  mj_func[IRP_MJ_DEVICE_CONTROL] = mx_mj_device_control;
  NdisMRegisterDevice(g_ndis_wrapper, &dev_name, &sym_name,
		      mj_func, &g_mxctl, &g_mxctl_handle);

  ndis_pnp = DriverObject->MajorFunction[IRP_MJ_PNP];
  DriverObject->MajorFunction[IRP_MJ_PNP] = mx_mj_pnp;

  MXE("<--- DriverEntry\n");
  return NDIS_STATUS_SUCCESS;

 abort_with_dev_lock:
  mx_spin_lock_destroy(&g_dev_lock);
  mx_finalize_driver();
 abort_with_wrapper:
  NdisTerminateWrapper(g_ndis_wrapper, NULL);

  return status;
}

void mx_ether_tx_done(struct mx_instance_state *is, uint32_t mcp_index)
{
  struct mx_ether *eth;
  int idx;
  struct mx_packetq_entry *entry;
  PNDIS_PACKET packet;

  //MXE("mx_ether_tx_done\n");

  eth = is->ether;
  NdisAcquireSpinLock(&eth->arch.adapter->send_lock);
  while (eth->tx.done != mcp_index) {
    idx = eth->tx.done & (NUM_TX - 1);
    packet = eth->tx.info[idx].packet;

    eth->tx.info[idx].packet = NULL;
    eth->tx.done++;

    if (packet != NULL) {
      eth->arch.adapter->xmit_ok++;
      //DbgPrint("packet %p sent\n", packet);
      NdisReleaseSpinLock(&eth->arch.adapter->send_lock);
      NdisMSendComplete(eth->arch.adapter->miniport_adapter_handle,
			packet, NDIS_STATUS_SUCCESS);
      NdisAcquireSpinLock(&eth->arch.adapter->send_lock);
    }
  }

  while (!STAILQ_EMPTY(&eth->arch.adapter->sendq)) {
    entry = STAILQ_FIRST(&eth->arch.adapter->sendq);
    STAILQ_REMOVE_HEAD(&eth->arch.adapter->sendq, entries);
    packet = CONTAINING_RECORD(entry, NDIS_PACKET,
			       MiniportReserved[sizeof (void*)]);
    if (mx_send_packet(eth->arch.adapter, packet, TRUE) !=
	NDIS_STATUS_SUCCESS) {
      /* mx_send_packet put it back at the head. Can change so that it
	 doesn't and we remove after successful send. */
      break;
    }
  }
  NdisReleaseSpinLock(&eth->arch.adapter->send_lock);
}

unsigned int g_tcp_good = 0;
unsigned int g_tcp_bad = 0;
unsigned int g_udp_good = 0;
unsigned int g_udp_bad = 0;

void mx_ether_rx_done_small(struct mx_instance_state *is, int count,
			    int len, int csum, int flags)
{
  struct mx_ether *eth;
  int idx;
  void *va;
  NDIS_PHYSICAL_ADDRESS pa;
  PNDIS_PACKET packet;
  PNDIS_BUFFER buffer;
  NDIS_TCP_IP_CHECKSUM_PACKET_INFO checksum;
  struct ether_header *eh;
  struct ip *ip;
  uint16_t c;

  eth = is->ether;
  NdisAcquireSpinLock(&eth->arch.adapter->recv_lock);
  idx = eth->rx_small.cnt & (NUM_RX - 1);
#if 0
  DbgPrint("mx_ether_rx_done_small idx = %d, cnt = %d\n",
	   idx, eth->rx_small.cnt);
#endif
  eth->rx_small.cnt++;
  va = eth->rx_small.info[idx].va;
  pa = eth->rx_small.info[idx].pa;
  packet = eth->rx_small.info[idx].packet;
  buffer = eth->rx_small.info[idx].buffer;
  NdisAdjustBufferLength(buffer, len);
  *MX_PACKET_IDX(packet) = MX_SMALL_PACKET;
  *MX_PACKET_PA(packet) = pa;

  if (is->arch.adapter->do_csum_offload) {
    eh = (struct ether_header*) ((char*)va + MX_MCP_ETHER_PAD);
    ip = (struct ip*)((char*)eh + sizeof (struct ether_header));
    checksum.Value = 0;
    if (ip->ip_p == 6 || ip->ip_p == 17) { /* TODO: Magic number. */
      c = in_pseudo(ip->ip_src, ip->ip_dst,
		    htonl(csum + ntohs(ip->ip_len) -
			  ((ip->ip_hlv & 0x0F) << 2) + ip->ip_p));
      c ^= 0xffff;
      
      if (ip->ip_p == 6) {
	if (c == 0) {
	  checksum.Receive.NdisPacketTcpChecksumSucceeded = 1;
	  g_tcp_good++;
	}
	else {
	  checksum.Receive.NdisPacketTcpChecksumFailed = 1;
	  g_tcp_bad++;
	}
      }
      else {
	if (c == 0) {
	  checksum.Receive.NdisPacketUdpChecksumSucceeded = 1;
	  g_udp_good++;
	}
	else {
	  checksum.Receive.NdisPacketUdpChecksumFailed = 1;
	  g_udp_bad++;
	}
      }
    }
    NDIS_PER_PACKET_INFO_FROM_PACKET(packet, TcpIpChecksumPacketInfo) =
      (void*)(ULONG_PTR)checksum.Value;
  }
#if MX_RETURN_RECV_PACKET
  NDIS_SET_PACKET_STATUS(packet, NDIS_STATUS_RESOURCES);
#else
  NDIS_SET_PACKET_STATUS(packet, NDIS_STATUS_SUCCESS);
#endif
  //DbgPrint("received packet %p\n", packet);
  //dump_packet(packet, 1);
  NdisReleaseSpinLock(&eth->arch.adapter->recv_lock);
  NdisMIndicateReceivePacket(eth->arch.adapter->miniport_adapter_handle,
			     &packet, 1);
  NdisAcquireSpinLock(&eth->arch.adapter->recv_lock);
#if MX_RETURN_RECV_PACKET
  mx_return_packet_helper(is->arch.adapter, packet);
#endif
  eth->arch.adapter->rcv_ok++;
  NdisReleaseSpinLock(&eth->arch.adapter->recv_lock);
}

void mx_ether_rx_done_big(struct mx_instance_state *is, int count,
			  int len, int csum, int flags)
{
  struct mx_ether *eth;
  int idx;
  void *va;
  NDIS_PHYSICAL_ADDRESS pa;
  PNDIS_PACKET packet;
  PNDIS_BUFFER buffer;
  NDIS_TCP_IP_CHECKSUM_PACKET_INFO checksum;
  struct ether_header *eh;
  struct ip *ip;
  uint16_t c;

  eth = is->ether;
  NdisAcquireSpinLock(&eth->arch.adapter->recv_lock);
  idx = eth->rx_big.cnt & (NUM_RX - 1);
#if 0
  DbgPrint("mx_ether_rx_done_big idx = %d, cnt = %d\n",
	   idx, eth->rx_big.cnt);
#endif
  eth->rx_big.cnt++;
  va = eth->rx_big.info[idx].va;
  pa = eth->rx_big.info[idx].pa;
  packet = eth->rx_big.info[idx].packet;
  buffer = eth->rx_big.info[idx].buffer;
  NdisAdjustBufferLength(buffer, len);
  *MX_PACKET_IDX(packet) = MX_BIG_PACKET;
  *MX_PACKET_PA(packet) = pa;

  if (is->arch.adapter->do_csum_offload) {
    eh = (struct ether_header*) ((char*)va + MX_MCP_ETHER_PAD);
    ip = (struct ip*)((char*)eh + sizeof (struct ether_header));
    checksum.Value = 0;
    if (ip->ip_p == 6 || ip->ip_p == 17) { /* TODO: Magic number. */
      c = in_pseudo(ip->ip_src, ip->ip_dst,
		    htonl(csum + ntohs(ip->ip_len) -
			  ((ip->ip_hlv & 0x0F) << 2) + ip->ip_p));
      c ^= 0xffff;
      
      if (ip->ip_p == 6) {
	if (c == 0) {
	  checksum.Receive.NdisPacketTcpChecksumSucceeded = 1;
	  g_tcp_good++;
	}
	else {
	  checksum.Receive.NdisPacketTcpChecksumFailed = 1;
	  g_tcp_bad++;
	}
      }
      else {
	if (c == 0) {
	  checksum.Receive.NdisPacketUdpChecksumSucceeded = 1;
	  g_udp_good++;
	}
	else {
	  checksum.Receive.NdisPacketUdpChecksumFailed = 1;
	  g_udp_bad++;
	}
      }
    }
    NDIS_PER_PACKET_INFO_FROM_PACKET(packet, TcpIpChecksumPacketInfo) =
      (void*)(ULONG_PTR)checksum.Value;
  }
#if MX_RETURN_RECV_PACKET
  NDIS_SET_PACKET_STATUS(packet, NDIS_STATUS_RESOURCES);
#else
  NDIS_SET_PACKET_STATUS(packet, NDIS_STATUS_SUCCESS);
#endif
  NdisReleaseSpinLock(&eth->arch.adapter->recv_lock);
  NdisMIndicateReceivePacket(eth->arch.adapter->miniport_adapter_handle,
			     &packet, 1);
  NdisAcquireSpinLock(&eth->arch.adapter->recv_lock);
#if MX_RETURN_RECV_PACKET
  mx_return_packet_helper(is->arch.adapter, packet);
#endif
  eth->arch.adapter->rcv_ok++;
  NdisReleaseSpinLock(&eth->arch.adapter->recv_lock);
}

void
mx_ether_link_change_notify(mx_instance_state_t *is)
{
  NDIS_STATUS indicate_status;

  if(is->ether == NULL) {
    return;
  }
  if (is->link_state) {
    indicate_status = NDIS_STATUS_MEDIA_CONNECT;
  }
  else {
    indicate_status = NDIS_STATUS_MEDIA_DISCONNECT;
  }
  NdisMIndicateStatus(is->ether->arch.adapter->miniport_adapter_handle,
		      indicate_status, 0, 0);
  NdisMIndicateStatusComplete(is->ether->arch.adapter->miniport_adapter_handle);
}

void
mx_spin_lock_init(mx_spinlock_t *s, mx_instance_state_t *is, 
		  int endpoint, char *string)
{
  KeInitializeSpinLock(&s->spin_lock);
}

void
mx_sync_init (mx_sync_t *s, mx_instance_state_t *is, int unique, char *str)
{
  ExInitializeFastMutex(&s->mutex);
#if 0
  KeInitializeEvent(&s->sleep_event, SynchronizationEvent, FALSE);
#else
  mx_init_sem(&s->sem);
#endif
}

void *
mx_kmalloc (size_t len, uint32_t flags)
{
  void *p;

  p = ExAllocatePool(NonPagedPool, len);
  if (p != NULL) {
    g_kmem_cnt++;
    g_kmem_len += len;
  }
  if (p && (flags & MX_MZERO)) {
    bzero(p, len);
  }
  return p;
}

void
mx_kfree (void *ptr)
{
  ExFreePool(ptr);
  if (ptr != NULL) {
    g_kmem_cnt--;
  }
}

void *mx_copy_pin(PMDL *mdl, mx_uaddr_t uaddr, size_t ammount,
		  LOCK_OPERATION operation)
{
  PVOID p = NULL;

  *mdl = IoAllocateMdl((void*)uaddr, (ULONG)ammount, FALSE, TRUE, NULL);
  if (*mdl != NULL) {
    __try {
      MmProbeAndLockPages(*mdl, UserMode, operation);
    }
    __except (EXCEPTION_EXECUTE_HANDLER) {
      IoFreeMdl(*mdl);
      return NULL;
    }
    p = MmGetSystemAddressForMdlSafe(*mdl, NormalPagePriority);
    if (p == NULL) {
      MmUnlockPages(*mdl);
      IoFreeMdl(*mdl);
    }
  }
  return p;
}

void mx_copy_unpin(PMDL mdl)
{
  MmUnlockPages(mdl);
  IoFreeMdl(mdl);
}

int mx_arch_copyin(mx_uaddr_t what, void *where, size_t ammount)
{
  int retval = 0;
  PMDL mdl;
  void *p;

  if (ammount > 0) {
    p = mx_copy_pin(&mdl, what, ammount, IoReadAccess);
    if (p != NULL) {
      RtlCopyMemory(where, p, ammount);
      mx_copy_unpin(mdl);
    }
    else {
      retval = EFAULT;
    }
  }
  return retval;
}

int mx_arch_copyout(void *what, mx_uaddr_t where, size_t ammount)
{
  int retval = 0;
  PMDL mdl;
  void *p;

  if (ammount > 0) {
    p = mx_copy_pin(&mdl, where, ammount, IoWriteAccess);
    if (p != NULL) {
      RtlCopyMemory(p, what, ammount);
      mx_copy_unpin(mdl);
    }
    else {
      retval = EFAULT;
    }
  }
  return retval;
}

void
mx_set_default_hostname(void)
{
  strcpy(mx_default_hostname, "localhost");
}

void
mx_spin(uint32_t usecs)
{
#if 1
  KeStallExecutionProcessor(usecs);
#else
  if (usecs <= 50) {
    KeStallExecutionoProcessor(usecs);
  }
  else {
    /* TODO: KeDelayExecutionThread? */
    NdisMSleep(usecs);
  }
#endif
}

void
mx_sync_reset (mx_sync_t *s)
{
#if 0
  KeClearEvent(&s->sleep_event);
#else
  mx_init_sem(&s->sem);
#endif
}


void
mx_sync_destroy(mx_sync_t *s)
{
}

void
mx_mutex_enter(mx_sync_t *s)
{
  ExAcquireFastMutex(&s->mutex);
}

void
mx_mutex_exit(mx_sync_t *s)
{
  ExReleaseFastMutex(&s->mutex);
}

void 
mx_assertion_failed (const char *assertion, int line, const char *file)
{
  DbgPrint("MX: assertion: <<%x>>  failed at line %d, file %s\n",
	   assertion, line, file);
}

ULONG RtlRandom(PULONG seed);
int
mx_rand(void)
{
  LARGE_INTEGER cur_time;
  KeQuerySystemTime(&cur_time);
  return (int)RtlRandom(&cur_time.u.LowPart);
}

void
mx_wake(mx_sync_t * s)
{
  /*DbgPrint("mx_wake\n");*/
#if 0
  KeSetEvent(&s->sleep_event, 0, FALSE);
#else
  KeReleaseSemaphore(&s->sem, 0, 1, FALSE);
#endif
}

int
mx_sleep(mx_sync_t *s, int ms, int flags)
{
  int ret;
  PLARGE_INTEGER ptimeout;
  LARGE_INTEGER timeout;
  KPROCESSOR_MODE wait_mode;
  NTSTATUS status;

  /*DbgPrint("mx_sleep\n");*/

  if (ms == MX_MAX_WAIT) {
    ptimeout = NULL;
  }
  else {
    timeout.QuadPart = ms;
    timeout.QuadPart *= 1000; /* to microseconds */
    timeout.QuadPart *= 10;   /* to 100 nanoseconds */
    timeout.QuadPart *= -1;   /* to relative time */
    ptimeout = &timeout;
  }
  if (flags & MX_SLEEP_INTR) {
    wait_mode = UserMode;
  }
  else {
    wait_mode = KernelMode;
  }

#if 0
  status = KeWaitForSingleObject(&s->sleep_event, Executive, wait_mode,
				 FALSE, ptimeout);
#else
  status = KeWaitForSingleObject(&s->sem, Executive, wait_mode,
				 FALSE, ptimeout);
#endif
  switch (status) {
  case STATUS_SUCCESS:
    ret = 0;
    break;
  case STATUS_ALERTED:
  case STATUS_USER_APC:
    ret = EINTR;
    break;
  case STATUS_TIMEOUT:
    ret = EAGAIN;
    break;
  }

  return ret;
}

/* OS specific callback for direct get, copying from another process
 * user-space to current process user-space.
 */
int
mx_arch_copy_user_to_user(mx_uaddr_t udst,
			  mx_uaddr_t usrc, void * src_space,
			  uint32_t len)
{
  return EFAULT;
}

int
mx_direct_get(mx_endpt_state_t *dst_es, mx_shm_seg_t *dst_segs, uint32_t dst_nsegs,
	      mx_endpt_state_t *src_es, mx_shm_seg_t *src_segs, uint32_t src_nsegs,
	      uint32_t length)
{
  return EFAULT;
}

int
mx_start_mapper(mx_instance_state_t *is)
{
  return 0;
}

int
mx_stop_mapper(mx_instance_state_t *is)
{
  return 0;
}

int
mx_ether_parity_detach(mx_instance_state_t *is)
{
  MX_WARN(("mx_ether_parity_detach called!\n"));
  return 0;
}

void
mx_ether_parity_reattach(mx_instance_state_t *is)
{
  MX_WARN(("mx_ether_parity_reattach called!\n"));
}

void *
mx_map_pci_space (mx_instance_state_t * is, int bar, uint32_t offset, uint32_t len)
{
  void *ptr = NULL;
  PHYSICAL_ADDRESS addr;

  if (bar == 0) {
    addr.QuadPart = is->arch.iomem_base.QuadPart + offset;
    ptr = MmMapIoSpace(addr, len, MmWriteCombined);
  }
  else if (bar == 2) {
    addr.QuadPart = is->arch.special_base.QuadPart + offset;
    ptr = MmMapIoSpace(addr, len, MmWriteCombined);
  }
  return ptr;
}


void
mx_unmap_io_space (mx_instance_state_t * is,
		   uint32_t len, void *kaddr)
{
  DbgPrint("mx_unmap_io_space\n");
  MmUnmapIoSpace(kaddr, len);
}

int
mx_pcie_link_reset(mx_instance_state_t *is)
{
  return mx_frob_pcie_link(is->arch.adapter->bus_num);
}


int
mx_write_pci_config(mx_instance_state_t *is, uint32_t offset,
		    void *value, unsigned long length)
{
  NdisWritePciSlotInformation(is->arch.miniportAdapterHandle,
			      0, offset, value, length);
  return 0;
}

int
mx_write_pci_config_8(mx_instance_state_t *is, uint32_t offset,
		       uint8_t value)
{
  return mx_write_pci_config(is, offset, &value, sizeof (value));
}

int
mx_write_pci_config_16(mx_instance_state_t *is, uint32_t offset,
		       uint16_t value)
{
  return mx_write_pci_config(is, offset, &value, sizeof (value));
}

int
mx_write_pci_config_32(mx_instance_state_t *is, uint32_t offset,
		       uint32_t value)
{
  return mx_write_pci_config(is, offset, &value, sizeof (value));
}

int
mx_read_pci_config(mx_instance_state_t *is, uint32_t offset,
		   void *value, unsigned long length)
{
  NdisReadPciSlotInformation(is->arch.miniportAdapterHandle,
			     0, offset, value, length);
  return 0;
}

int
mx_read_pci_config_8(mx_instance_state_t *is, uint32_t offset,
		     uint8_t *value)
{
  return mx_read_pci_config(is, offset, value, sizeof (*value));
}

int
mx_read_pci_config_16(mx_instance_state_t *is, uint32_t offset,
		      uint16_t *value)
{
  return mx_read_pci_config(is, offset, value, sizeof (*value));
}

int
mx_read_pci_config_32(mx_instance_state_t *is, uint32_t offset,
		      uint32_t *value)
{
  return mx_read_pci_config(is, offset, value, sizeof (*value));
}
